# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox import common
from sandbox.projects.yabs.auto_supbs_2.lib.distribution_issue_processor import AutoSupBsDistributionIssueProcessor
from sandbox.projects.yabs.auto_supbs_2.lib.simple_issue_processor import AutoSupBsSimpleIssueProcessor
from sandbox.projects.yabs.auto_supbs_2.lib.matching_funnel_issue_processor import MatchingFunnelProcessYQLResult
from sandbox.projects.yabs.auto_supbs_2.lib.audit_issue_processor import (
    AuditDailyBudgetVersionedIssueProcessor,
    AuditBudgetVersionedIssueProcessor,
    AuditCPCVersionedIssueProcessor,
    AuditCPMVersionedIssueProcessor,
    AuditFixCPMIssueProcessor,
    AuditAnyBudgetIssueProcessor,
    AuditCpcIssueProcessor,
    AuditCpmIssueProcessor,
    AuditBidsIssueProcessor,
    AuditBidsClosedIssueProcessor
)
from sandbox.projects.yabs.base_bin_task import BaseBinTask

import sandbox.common.types.task as ctt


class AutoSupBsQueue2(BaseBinTask):

    class Parameters(BaseBinTask.Parameters):
        with sdk2.parameters.String('Issue type', default='simple_autosupbs') as autosupbs_issue_type:
            for issue_type in ['simple_autosupbs', 'distribution_autosupbs', 'audit_daily_budget',
                               'audit_daily_budget_week', 'audit_cpc', 'audit_cpm', 'audit_budget', 'audit_bids', 'matching_funnel_run_yql',
                               'matching_funnel_process_yql_result']:
                autosupbs_issue_type.values[issue_type] = issue_type

        startrek_token = sdk2.parameters.String(
            'startrek token sandbox vault name',
            default='ROBOT_AUTOSUPBS_ST_TOKEN',
            required=True
        )

        yt_token = sdk2.parameters.String(
            'yt token sandbox vault name',
            default='ROBOT_AUTOSUPBS_YT_TOKEN',
            required=True
        )

        yql_token = sdk2.parameters.String(
            'ql token sandbox vault name',
            default='ROBOT_AUTOSUPBS_YQL_TOKEN',
            required=True
        )

        startrek_filter_id = sdk2.parameters.String(
            'startrek filter id',
            default='11229',
            required=True
        )

        remove_tag_after_processing = sdk2.parameters.Bool(
            'Remove tag after processing',
            default=False,
        )

        with remove_tag_after_processing.value[True]:
            tag_to_remove = sdk2.parameters.String(
                "Tag to remove after processing",
            )

        add_tag_after_processing = sdk2.parameters.Bool(
            'Add new tag after processing',
            default=False,
        )

        with add_tag_after_processing.value[True]:
            tag_to_add = sdk2.parameters.String(
                "Tag to add after processing",
            )

        startrek_api_url = sdk2.parameters.String(
            'startrek api url',
            default='https://st-api.test.yandex-team.ru',
            required=True
        )

        ticket_example = sdk2.parameters.String(
            'example of correct parameters filling',
            default='',
            required=False
        )

    def on_execute(self):
        from startrek_client import Startrek

        autosupbs_issue_type = self.Parameters.autosupbs_issue_type
        startrek_token = sdk2.Vault.data(self.owner, self.Parameters.startrek_token)
        startrek_api_url = self.Parameters.startrek_api_url
        startrek_filter_id = self.Parameters.startrek_filter_id
        remove_tag_after_processing = self.Parameters.remove_tag_after_processing
        tag_to_remove = self.Parameters.tag_to_remove
        add_tag_after_processing = self.Parameters.add_tag_after_processing
        tag_to_add = self.Parameters.tag_to_add
        ticket_example = self.Parameters.ticket_example
        yt_token = self.Parameters.yt_token
        yql_token = self.Parameters.yql_token

        startrek = Startrek(
            useragent='autosupbs',
            base_url=startrek_api_url,
            token=startrek_token
        )
        issues = startrek.issues.find(
            filter_id=startrek_filter_id
        )
        tickets = [issue.key for issue in issues]

        if self.Context.tickets:
            if sdk2.Task[self.Context.sub_task_id].status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure('Subtask failed')
            task_result = 'The following tickets were processed:\n{}'.format(
                ', '.join(self.Context.tickets)
            )
            self.set_info(task_result)
        elif tickets:
            logging.info('Tickets to process:')
            logging.info(tickets)
            with self.memoize_stage.create_children:
                if autosupbs_issue_type == 'distribution_autosupbs':
                    logging.info("Creating Autosupbs Distribution task")
                    sub_task = AutoSupBsDistributionIssueProcessor(
                        self,
                        tickets=tickets,
                        create_sub_task=False,
                        description='Child of task {}'.format(self.id),
                        startrek_api_url=startrek_api_url,
                        startrek_token=self.Parameters.startrek_token,
                        remove_tag_after_processing=remove_tag_after_processing,
                        tag_to_remove=tag_to_remove,
                        ticket_example=ticket_example,
                    )
                elif autosupbs_issue_type == 'simple_autosupbs':
                    logging.info("Creating Simple Autosupbs task")
                    sub_task = AutoSupBsSimpleIssueProcessor(
                        self,
                        tickets=tickets,
                        create_sub_task=False,
                        description='Child of task {}'.format(self.id),
                        startrek_api_url=startrek_api_url,
                        startrek_token=self.Parameters.startrek_token,
                        remove_tag_after_processing=remove_tag_after_processing,
                        tag_to_remove=tag_to_remove
                    )
                elif autosupbs_issue_type == 'matching_funnel_process_yql_result':
                    logging.info("Creating MatchingFunnelProcessYQLResult task")
                    sub_task = MatchingFunnelProcessYQLResult(
                        self,
                        tickets=tickets,
                        create_sub_task=False,
                        description='Child of task {}'.format(self.id),
                        startrek_api_url=startrek_api_url,
                        startrek_token=self.Parameters.startrek_token,
                        add_tag_after_processing=add_tag_after_processing,
                        tag_to_add=tag_to_add,
                        yt_token=yt_token,
                        yql_token=yql_token
                    )
                elif autosupbs_issue_type.startswith('audit'):
                    common_params = {
                        'tickets': tickets,
                        'create_sub_task': False,
                        'description': 'Child of task {}'.format(self.id),
                        'startrek_api_url': startrek_api_url,
                        'startrek_token': self.Parameters.startrek_token,
                        'remove_tag_after_processing': remove_tag_after_processing,
                        'tag_to_remove': tag_to_remove,
                        'summon_assignee': True,
                    }
                    if autosupbs_issue_type == 'audit_budget_versioned':
                        sub_task = AuditBudgetVersionedIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_daily_budget_versioned':
                        sub_task = AuditDailyBudgetVersionedIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_daily_budget_week_versioned':
                        sub_task = AuditBudgetVersionedIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_cpc_versioned':
                        sub_task = AuditCPCVersionedIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_cpm_versioned':
                        sub_task = AuditCPMVersionedIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_fix_cpm':
                        sub_task = AuditFixCPMIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_daily_budget':
                        sub_task = AuditAnyBudgetIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_daily_budget_week':
                        sub_task = AuditAnyBudgetIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_cpc':
                        sub_task = AuditCpcIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_cpm':
                        sub_task = AuditCpmIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_budget':
                        sub_task = AuditAnyBudgetIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_bids':
                        common_params['tickets'] = common_params['tickets'][:1]
                        sub_task = AuditBidsIssueProcessor(self, **common_params)
                    elif autosupbs_issue_type == 'audit_bids_closed':
                        sub_task = AuditBidsClosedIssueProcessor(self, **common_params)

                sub_task.Requirements.tasks_resource = self.Requirements.tasks_resource
                sub_task.save().enqueue()
                self.Context.sub_task_id = sub_task.id
                self.Context.tickets = tickets
                raise sdk2.WaitTask(
                    sub_task.id,
                    common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
                    wait_all=True
                )
        else:
            self.set_info('Nothing to process')
