# -*- coding: utf-8 -*-

import logging
import re

from sandbox import sdk2
from sandbox.projects.yabs.auto_supbs_2.lib.order_processor_lib import OrderProcessor

from sandbox.projects.yabs.auto_supbs_2.common import (
    BaseAutoSupBsIssueProcessor,
    TicketError,
)


class LookupProfileDump(sdk2.Resource):
    """ Dump of lookup_profile binary """


class AutoSupBsSimpleIssueProcessor(BaseAutoSupBsIssueProcessor):

    class Parameters(BaseAutoSupBsIssueProcessor.Parameters):
        yt_token = sdk2.parameters.String(
            'yt token sandbox vault name',
            default='ROBOT_AUTOSUPBS_YT_TOKEN',
            required=True
        )
        yql_token = sdk2.parameters.String(
            'yql token sandbox vault name',
            default='ROBOT_AUTOSUPBS_YQL_TOKEN',
            required=True
        )
        charts_token = sdk2.parameters.String(
            'yql token sandbox vault name',
            default='ROBOT_AUTOSUPBS_CHARTS_TOKEN',
            required=True
        )
        caesar_adgroups_path = sdk2.parameters.String(
            'Caesar_adgroups_path',
            default='//home/bs/logs/AdsCaesarAdGroupsFullDump/latest',
            required=True
        )
        graph_params_and_percentiles = sdk2.parameters.JSON(
            'Parameters which will be shown on graphs and their percentiles',
            default={
                'BidCorrection': [10, 50, 90],
                'ABConversionCostCoef': [10, 50, 90],
                'EventCost': [10, 50, 90],
                'EventCostCur': [10, 50, 90],
                'Rank': [10, 50, 90]
            },
            required=True
        )
        caesar_max_entity_count = sdk2.parameters.Integer(
            'Limit to the number of entities to view in single comment',
            default=100,
            required=True
        )

        lookup_profile_dump_ttl = sdk2.parameters.Integer('TTL for CaeSaR Dump Files', default_value=60)

    def process_issue(self, issue, comments):
        logging.info('Processing {}'.format(issue.key))

        if issue.description is None:
            raise TicketError('Empty Description')
        orders = set(re.findall(
            u'БК:\**\s*\**(\d+)\**\s*',  # noqa
            issue.description,
            re.UNICODE
        ))
        if not orders:
            raise TicketError('OrderID was not found in the Description')
        graph_params = self.Parameters.graph_params_and_percentiles

        order_processor = OrderProcessor(
            sdk2.Vault.data(self.owner, self.Parameters.yt_token),
            self.Parameters.caesar_adgroups_path,
            sdk2.Vault.data(self.owner, 'ROBOT_AUTOSUPBS_YQL_TOKEN_TEMPORARY'),
            sdk2.Vault.data(self.owner, self.Parameters.charts_token),
            issue.start,
            issue.end,
            params=graph_params
        )

        logging.info('The following orders were found {}'.format(orders))

        attributes = {'ttl': self.Parameters.lookup_profile_dump_ttl}
        lookup_profile_resource = LookupProfileDump(
            task=self,
            description="CaeSaR Requests full dump in {}".format(str(issue.key)),
            path=str(issue.key),
            **attributes
        )

        for order_id in orders:
            try:
                comments.append(order_processor.build_graphs(order_id))
            except Exception:
                error_msg = "Error while Building Graph for OrderID {} in ((https://sandbox.yandex-team.ru/task/{}/view SanboxTask))".format(order_id, self.id)
                comments.append([error_msg])
                logging.error(error_msg, exc_info=True)

            try:
                lookup_profile_data = sdk2.ResourceData(lookup_profile_resource)
                lookup_profile_data.path.mkdir(0o755, parents=True, exist_ok=True)

                messages, lookup_profile_dump_text = order_processor.process_order_from_caesar(
                    order_id=order_id,
                    task_id=self.id,
                    max_entity_count=self.Parameters.caesar_max_entity_count
                )

                lookup_profile_data.path.joinpath("order_{}.txt".format(order_id)).write_bytes("\n\n".join(lookup_profile_dump_text))
                lookup_profile_data.ready()

                messages.append("\n")
                messages.append(
                    "CaeSaR Requests full dump can be found here: (({}/order_{}.txt))".format(lookup_profile_resource.http_proxy, order_id)
                )
                comments.append(messages)
            except Exception:
                error_msg = "Error while executing CaeSaR Requests for OrderID {} in ((https://sandbox.yandex-team.ru/task/{}/view SanboxTask))".format(order_id, self.id)
                comments.append([error_msg])
                logging.error(error_msg, exc_info=True)

            try:
                messages = order_processor.process_order_genocide(
                    order_id=order_id,
                    task_id=self.id,
                    max_entity_count=self.Parameters.caesar_max_entity_count
                )

                comments.append(messages)
            except Exception:
                error_msg = "Error while executing Genocide Requests for OrderID {} in ((https://sandbox.yandex-team.ru/task/{}/view SanboxTask))".format(order_id, self.id)
                comments.append([error_msg])
                logging.error(error_msg, exc_info=True)
