from __future__ import print_function
import os
import time
import logging

import requests

from sandbox import sdk2

from sandbox.projects.yabs.autocampaign.utils.sandbox_tools import download_resource_file
from sandbox.sdk2.helpers import subprocess


class YABS_AUTOCAMPAIGN_UPLOAD_RESULT(sdk2.Resource):
    """ tsv-file with autocampaign uploading result """
    any_arch = True
    auto_backup = True
    executable = False


class YabsAutocampaignUploader(sdk2.Task):
    """ Upload ads to Direct API """
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        description = "Uploading ads for autocampaigns %s" % time.strftime("%Y-%m-%d %H:%M:%S")

        with sdk2.parameters.Group("Upload parameters") as upload_parameters:
            input_table_path = sdk2.parameters.String("Input pable path",
                                                      default="//home/yabs/autocampaign/input/postprocessing")

            direct_login = sdk2.parameters.String("Direct login",
                                                  default="turkish-free-api")

            skip_check_hold = sdk2.parameters.Bool("Don't check manually published campaigns", default=False)
            ignore_uploaded = sdk2.parameters.Bool("Upload data even if it was marked as uploaded", default=False)

        with sdk2.parameters.Group("Vault") as vault:
            yt_token = sdk2.parameters.String(
                label="YT token vault name",
                name="yt_token",
                default="YABS_AUTOCAMPAIGN_YT_TOKEN",
                required=True,
            )

            yql_token = sdk2.parameters.String(
                label="YQL token vault name",
                name="yt_token",
                default="YABS_AUTOCAMPAIGN_YQL_TOKEN",
                required=True,
            )

            direct_token = sdk2.parameters.String(
                label="Direct token vault name",
                name="direct_token",
                default="YABS_AUTOCAMPAIGN_DIRECT_TOKEN",
                required=True,
            )

        with sdk2.parameters.Group("Testing") as testing:
            testing_mode = sdk2.parameters.Bool(
                label="Testing mode",
                name="testing_mode",
                default=False
            )

            with testing_mode.value[True]:
                stats_yt_proxy = sdk2.parameters.String("stats_yt_proxy",
                                                        default="igorock.haze.yandex.net:15562")
                input_yt_proxy = sdk2.parameters.String("input_yt_proxy",
                                                        default="igorock.haze.yandex.net:15562")

                direct_api_host = sdk2.parameters.String("direct_api_host",
                                                         default="ipv6.test-direct.yandex.ru")
                direct_api_port = sdk2.parameters.Integer("direct_api_port",
                                                          default=14443)

    def save_result(self, filename, description=""):
        if os.path.isfile(filename):
            resource = YABS_AUTOCAMPAIGN_UPLOAD_RESULT(self, description, filename, ttl=30)

            sdk2.ResourceData(resource).ready()
            resp = requests.get(resource.url)
            if resp:
                url = resp.json()["http"]["proxy"]
                logging.info("File %s uploaded to %s" % (filename, url))
                return filename, url
            else:
                return filename, None
        else:
            return None, None

    def attach_output(self, output, links=()):
        info = ""
        header_template = """<div class="title" style="font-family: Arial, Helvetica, sans-serif;color: #666;font-size: 16px;font-weight: bold;padding: 15px 0 15px 0">%s</div>"""
        info += """%(raw_output_header)s
        <pre>%(raw_output)s</pre>
        """ % {"raw_output_header": header_template % "Raw output", "raw_output": output}

        for filename, link in links:
            if link is not None:
                info += "<a href=\"%(link)s\" style=\"color: #127ADB; text-decoration: none;\">%(filename)s</a>" % {
                    "filename": filename,
                    "link": link
                }
        logging.debug(info)

        self.set_info(info, do_escape=False)

    def run(self):
        ad_stencil2 = download_resource_file(resource_name="ad_stencil2")
        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        yql_token = sdk2.Vault.data(self.Parameters.yql_token)
        direct_token = sdk2.Vault.data(self.Parameters.direct_token)

        parsing_failed_file = "parsing_failed.tsv"

        os.environ["YT_TOKEN"] = yt_token
        os.environ["YQL_TOKEN"] = yql_token
        os.environ["DIRECT_TOKEN"] = direct_token

        command = [
            ad_stencil2,
            "upload",
            "--direct-login %s" % self.Parameters.direct_login,
            "--input-table %s" % self.Parameters.input_table_path,
            "--parsing-failed-file %s" % parsing_failed_file,
        ]

        if self.Parameters.skip_check_hold:
            command.append("--skip-check-hold")
        if self.Parameters.ignore_uploaded:
            command.append("--ignore-uploaded")

        if self.Parameters.testing_mode:
            command += [
                "--input-yt-proxy %s" % self.Parameters.input_yt_proxy,
                "--stats-yt-proxy %s" % self.Parameters.stats_yt_proxy,
                "--direct-api-host %s" % self.Parameters.direct_api_host,
                "--direct-api-port %d" % self.Parameters.direct_api_port,
            ]

        with sdk2.helpers.ProcessLog(self, logger="ad_stencil") as pl:
            pl.logger.propagate = 1
            res = subprocess.check_output(str(" ".join(command)), shell=True, stderr=pl.stderr)

        links = []
        links.append(self.save_result(parsing_failed_file, "Parsing failed"))

        self.attach_output(res, links=links)

    def on_execute(self):
        self.run()
