import logging


BATCH_SIZE = 1000000
YT_SCHEMA = [
    {"required": True, "type": "string", "name": "Time"},
    {"required": True, "type": "uint32", "name": "Action"},
    {"required": False, "type": "uint32", "name": "SectionNmb"},
    {"required": False, "type": "uint32", "name": "SubSectionNmb"},
    {"required": False, "type": "uint32", "name": "PlacementNmb"},
    {"required": False, "type": "uint32", "name": "AdNmb"},
    {"required": False, "type": "uint64", "name": "UserNmb"},
    {"required": False, "type": "boolean", "name": "KnownUser"},
    {"required": False, "type": "string", "name": "Referrer"},
    {"required": False, "type": "string", "name": "UserAgent"},
    {"required": False, "type": "string", "name": "ParameterStr"},
    {"required": False, "type": "string", "name": "IP"},
    {"required": False, "type": "uint32", "name": "PageLoadId"},
    {"required": False, "type": "uint32", "name": "LogNmb"},
    {"required": False, "type": "uint32", "name": "GeoNmb"},
    {"required": False, "type": "uint64", "name": "GlobalRequestId"},
    {"required": False, "type": "uint8", "name": "SdGenderM"},
    {"required": False, "type": "uint8", "name": "SdGenderF"},
    {"required": False, "type": "uint8", "name": "SdAge0"},
    {"required": False, "type": "uint8", "name": "SdAge18"},
    {"required": False, "type": "uint8", "name": "SdAge25"},
    {"required": False, "type": "uint8", "name": "SdAge35"},
    {"required": False, "type": "uint8", "name": "SdAge45"},
    {"required": False, "type": "uint8", "name": "SdAge45_55"},
    {"required": False, "type": "uint8", "name": "SdAge56_plus"},
    {"required": False, "type": "uint8", "name": "SdIncomeA"},
    {"required": False, "type": "uint8", "name": "SdIncomeB"},
    {"required": False, "type": "uint8", "name": "SdIncomeC"},
    {"required": False, "type": "uint8", "name": "SdVersion"},
    {"required": False, "type": "int32", "name": "RtbHostId"},
    {"required": False, "type": "int32", "name": "AdplaceNmb"},
    {"required": False, "type": "int32", "name": "RtbStlmPrice"},
    {"required": False, "type": "int32", "name": "DynamicPosition"},
    {"required": False, "type": "int32", "name": "AdElementId"},
]


class YabsAwapsAuditLogExportHandler():
    def __init__(self, yt_cluster, yt_path, yt_token, sql_host, sql_username, sql_password):
        self.yt_cluster = yt_cluster
        self.yt_path = yt_path
        self.yt_token = yt_token

        self.sql_host = sql_host
        self.sql_username = sql_username
        self.sql_password = sql_password

    def export(self, log_date):
        import yt.wrapper as yt
        from sqlalchemy import create_engine, MetaData, Table
        from sqlalchemy.engine.reflection import Inspector
        from sqlalchemy.orm import sessionmaker

        def long2ip(ip):
            d = (ip & (0xff << 24)) >> 24
            c = (ip & (0xff << 16)) >> 16
            b = (ip & (0xff << 8)) >> 8
            a = ip & 0xff
            return "%s.%s.%s.%s" % (a, b, c, d)

        def isnull(value, default):
            return default if value is None else value

        yt_table = "{path}/{date:%Y-%m-%d}".format(path=self.yt_path, date=log_date)
        sql_database = "day_logs_{date:%y_%m_%d}".format(date=log_date)  # day_logs_20_09_29

        logging.info("{host}/{db} -> {yt}".format(host=self.sql_host, db=sql_database, yt=yt_table))

        yt_client = yt.YtClient(self.yt_cluster, token=self.yt_token)
        sql_engine = create_engine(
            'mssql+pymssql://{username}:{password}@{host}/{database}'.format(host=self.sql_host,
                                                                             database=sql_database,
                                                                             username=self.sql_username,
                                                                             password=self.sql_password))
        SessionMaker = sessionmaker(bind=sql_engine)
        session = SessionMaker()
        try:
            sql_inspector = Inspector.from_engine(sql_engine)
            table_ActionCampaignSl = Table("ActionCampaignSl", MetaData())
            sql_inspector.reflecttable(table_ActionCampaignSl, None)
            query = session.query(table_ActionCampaignSl)

            if yt_client.exists(yt_table):
                yt_client.remove(yt_table, recursive=True)

            yt_client.create('table', yt_table, recursive=True, attributes={
                'schema': YT_SCHEMA,
            })

            yt_rows_sent = 0
            yt_batch = []
            for r in query.yield_per(BATCH_SIZE):
                yt_row = {
                    'Time': r.Time.strftime('%Y-%m-%dT%H:%M:%S'),
                    'Action': int(r.Action),
                    'SectionNmb': r.SectionNmb,
                    'SubSectionNmb': r.SubSectionNmb,
                    'PlacementNmb': r.PlacementNmb,
                    'AdNmb': r.AdNmb,
                    'UserNmb': int(r.UserNmb),
                    'Referrer': r.Referrer,
                    'UserAgent': r.UserAgent,
                    'KnownUser': r.KnownUser,
                    'IP': long2ip(r.IP),
                    'PageLoadId': r.PageLoadID,
                    'LogNmb': r.logNmb,
                    'GeoNmb': r.GeoNmb,
                    'GlobalRequestId': r.global_request_id,
                    'SdGenderM': isnull(r.sd_gender_m, 0),
                    'SdGenderF': isnull(r.sd_gender_f, 0),
                    'SdAge0': isnull(r.sd_age_0, 0),
                    'SdAge18': isnull(r.sd_age_18, 0),
                    'SdAge25': isnull(r.sd_age_25, 0),
                    'SdAge35': isnull(r.sd_age_35, 0),
                    'SdAge45': isnull(r.sd_age_45, 0),
                    'SdAge45_55': isnull(r.sd_age_45_55, 0),
                    'SdAge56_plus': isnull(r.sd_age_56_plus, 0),
                    'SdIncomeA': isnull(r.sd_income_a, 0),
                    'SdIncomeB': isnull(r.sd_income_b, 0),
                    'SdIncomeC': isnull(r.sd_income_c, 0),
                    'RtbHostId': isnull(r.rtb_host_id, 0),
                    'ParameterStr': isnull(r.parameterStr, ''),
                    'AdplaceNmb': isnull(r.adplace_nmb, 0),
                    'RtbStlmPrice': r.rtb_stlm_price,
                    'DynamicPosition': r.dynamic_position,
                    'AdElementId': r.ad_element_id,
                    # 'IPv6': str(r.ipv6),
                    'SdVersion': r.sd_version,
                }
                yt_batch.append(yt_row)

                if len(yt_batch) == BATCH_SIZE:
                    yt_client.write_table(yt.TablePath(yt_table, append=(yt_rows_sent > 0)),
                                          yt_batch,
                                          format=yt.JsonFormat(attributes={"encode_utf8": False}, encoding='utf-8'))
                    yt_rows_sent += len(yt_batch)
                    yt_batch[:] = []  # clear list
                    logging.info("{rows} rows sent".format(rows=yt_rows_sent))

            if len(yt_batch) > 0:
                yt_client.write_table(yt.TablePath(yt_table, append=(yt_rows_sent > 0)),
                                      yt_batch,
                                      format=yt.JsonFormat(attributes={"encode_utf8": False}, encoding='utf-8'))
                yt_rows_sent += len(yt_batch)
                logging.info("{rows} rows sent".format(rows=yt_rows_sent))

        finally:
            session.close()
