import logging


THRESHOLD_RATE = 1.02
THRESHOLD_SHOWS = 100
THRESHOLD_RATE_TODAY = 1.01
THRESHOLD_SHOWS_TODAY = 100
THRESHOLD_RATE_MIN = 0.5
THRESHOLD_SHOWS_MIN = 1000

CASSANDRA_NODES = [
    'awaps-cassandra01e.yandex.ru',
    'awaps-cassandra01f.yandex.ru',
    'awaps-cassandra01i.yandex.ru',
    'awaps-cassandra01v.yandex.ru',
    'awaps-cassandra02e.yandex.ru',
    'awaps-cassandra02f.yandex.ru',
    'awaps-cassandra02i.yandex.ru',
    'awaps-cassandra02v.yandex.ru',
    'awaps-cassandra03e.yandex.ru',
    'awaps-cassandra03f.yandex.ru',
    'awaps-cassandra03i.yandex.ru',
    'awaps-cassandra03v.yandex.ru',
    'awaps-cassandra04e.yandex.ru',
    'awaps-cassandra04f.yandex.ru',
    'awaps-cassandra04i.yandex.ru',
    'awaps-cassandra04v.yandex.ru',
    'awaps-cassandra05e.yandex.ru',
    'awaps-cassandra05f.yandex.ru',
    'awaps-cassandra05i.yandex.ru',
    'awaps-cassandra05v.yandex.ru',
    'awaps-cassandra06e.yandex.ru',
    'awaps-cassandra06f.yandex.ru',
    'awaps-cassandra06i.yandex.ru',
    'awaps-cassandra06v.yandex.ru',
]

QUERY_AW_MASTER_ACTIVE_PLACEMENTS = """
;with stat as (
select PlacementNmb, sum(CountNmb) as shows
    from em_logs.dbo.ActionCountHour with(nolock)
    where Action = 0 and PlacementNmb > 30000
    group by PlacementNmb
    having sum(CountNmb) > 0
)
select p.xm_order_nmb as flight_nmb, xm_placement_key as placement_id,
        c.xm_PLANNED_SHOW_COUNT as planned_shows, p.xm_CURRENT_SHOW_COUNT as current_shows,
        (SELECT SUM(budget) FROM v_placement_budget_spent with(nolock) WHERE xm_placement_key = p.xm_placement_key) as spent_budget,
        cast(p.xm_BEGIN_DATE as date) as date_begin, dateadd(d, -1, cast(p.xm_END_DATE as date)) as date_end, datediff(d, cast(getdate() as date), cast(p.xm_END_DATE as date)) as days_left
    from v_placement_extended p with(nolock)
        join stat s on s.PlacementNmb = p.xm_PLACEMENT_KEY
        join v_campaign c on c.m_id = p.xm_CAMPAIGN_ID
    where p.xm_order_nmb is not null
        and (p.xm_CURRENT_SHOW_COUNT > 10000 or (p.xm_CURRENT_SHOW_COUNT > 1000 and c.xm_PLANNED_SHOW_COUNT < 100000))
"""

QUERY_CASSANDRA_PLACEMENT_COUNTERS = 'SELECT "PlacementId", "CurrentShowCount", "TotalPayments" FROM "PlacementCounter" WHERE "PlacementId" IN %s and "ExperimentId" = 0'


class YabsAwapsCassandraAdjust():
    def __init__(self, sql_host, sql_username, sql_password):
        self.sql_host = sql_host
        self.sql_username = sql_username
        self.sql_password = sql_password
        self.sql_database = 'aw_master'

    def process(self, mode):  # , placementid, threshold
        from sqlalchemy import create_engine
        from cassandra.cluster import Cluster
        from cassandra import query
        from cassandra import ConsistencyLevel
        from cassandra.policies import RetryPolicy

        def get_connection_string(host, username, password, database=''):
            return 'mssql+pymssql://{username}:{password}@{host}/{database}'\
                .format(host=host, username=username, password=password, database=database)

        active_placements = {}
        flight_placements = {}
        sql_engine = create_engine(get_connection_string(self.sql_host, self.sql_username, self.sql_password, self.sql_database))
        with sql_engine.connect() as sql_connection:
            result = sql_connection.execute(QUERY_AW_MASTER_ACTIVE_PLACEMENTS)
            for row in result:
                active_placements[row['placement_id']] = {
                    'flight_nmb': int(row['flight_nmb']),
                    'planned_shows': row['planned_shows'],
                    'current_shows': row['current_shows'],
                    'spent_budget': row['spent_budget'],
                    'date_begin': row['date_begin'],
                    'date_end': row['date_end'],
                    'days_left': row['days_left'],
                }
                if int(row['flight_nmb']) not in flight_placements:
                    flight_placements[int(row['flight_nmb'])] = []
                flight_placements[int(row['flight_nmb'])].append(row['placement_id'])

        cluster = Cluster(contact_points=CASSANDRA_NODES, connect_timeout=60, default_retry_policy=RetryPolicy.RETRY_NEXT_HOST)
        session = None
        for retry in range(5):
            logging.info('trying to connect Cassandra... attempt {retry}'.format(retry=retry))
            try:
                session = cluster.connect('pldata', wait_for_all_pools=True)
                logging.info('connected')
                break
            except:
                if retry == 4:
                    raise
                pass

        session.row_factory = query.dict_factory
        session.default_consistency_level = ConsistencyLevel.ALL

        cassandra_placements = {}
        for retry in range(5):
            logging.info('trying to select Cassandra placements... attempt {retry}'.format(retry=retry))
            try:
                result = session.execute(QUERY_CASSANDRA_PLACEMENT_COUNTERS, parameters=[query.ValueSequence(active_placements.keys())], timeout=120)
                for row in result:
                    cassandra_placements[row['PlacementId']] = {
                        'current_show_count': row['CurrentShowCount'],
                        'total_payments': row['TotalPayments'],
                    }
                    logging.info('done')
                break
            except:
                if retry == 4:
                    raise
                pass

        for placement_id, cassandra_data in cassandra_placements.items():
            aw_master_data = active_placements[placement_id]
            # is_quoted = len(flight_placements[aw_master_data['flight_nmb']]) > 1

            is_bad = False
            if aw_master_data['days_left'] == 1 \
                    and float(cassandra_data['current_show_count']) / aw_master_data['current_shows'] > THRESHOLD_RATE_TODAY \
                    and cassandra_data['current_show_count'] - aw_master_data['current_shows'] > THRESHOLD_SHOWS_TODAY:
                is_bad = True
            if aw_master_data['days_left'] > 1 \
                    and float(cassandra_data['current_show_count']) / aw_master_data['current_shows'] > THRESHOLD_RATE \
                    and cassandra_data['current_show_count'] - aw_master_data['current_shows'] > THRESHOLD_SHOWS:
                is_bad = True
            if aw_master_data['days_left'] > 1 \
                    and float(cassandra_data['current_show_count']) / aw_master_data['current_shows'] > THRESHOLD_RATE_MIN \
                    and cassandra_data['current_show_count'] + THRESHOLD_SHOWS_MIN < aw_master_data['current_shows']:
                is_bad = True

            logging.info(
                placement_id, aw_master_data['flight_nmb'],
                aw_master_data['current_shows'], cassandra_data['current_show_count'],
                aw_master_data['spent_budget'], cassandra_data['total_payments'],
                (float(cassandra_data['current_show_count']) / aw_master_data['current_shows']),
                aw_master_data['days_left'], is_bad)

            if not is_bad:
                continue

            diff_shows = aw_master_data['current_shows'] - cassandra_data['current_show_count']
            diff_payment = (aw_master_data['spent_budget'] - cassandra_data['total_payments']) if cassandra_data['total_payments'] is not None else 0

            if aw_master_data['days_left'] > 3 and diff_shows < 0:
                diff_shows = int(diff_shows / 2)
                diff_payment = int(diff_payment / 2)

            if cassandra_data['total_payments'] is not None and (aw_master_data['spent_budget'] < cassandra_data['total_payments']):
                update_query = 'UPDATE "PlacementCounter" ' \
                               '   SET "CurrentShowCount" = "CurrentShowCount" {diff_shows}, "TotalPayments" = "TotalPayments" {diff_payment} ' \
                               '   WHERE "PlacementId" = {placement} and "ExperimentId" = 0'.format(
                                   diff_shows=('+' + str(diff_shows)) if diff_shows > 0 else str(diff_shows),
                                   diff_payment=('+' + str(diff_payment)) if diff_payment > 0 else str(diff_payment),
                                   placement=placement_id)
            else:
                update_query = 'UPDATE "PlacementCounter" SET "CurrentShowCount" = "CurrentShowCount" {diff_shows} WHERE "PlacementId" = {placement} and "ExperimentId" = 0'.format(
                    diff_shows=('+' + str(diff_shows)) if diff_shows > 0 else str(diff_shows), placement=placement_id)
            logging.info(update_query)
            try:
                if mode == 'adjust':
                    session.execute(update_query, timeout=60)
                    logging.info('FIXED')
                else:
                    logging.info('CHECK')
            except:
                logging.info('EXCEPTION')

        session.shutdown()
