#!/usr/bin/env python
# -*- coding: utf-8 -*-
from datetime import datetime, date
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.sandboxsdk import environments
import boto3
import json
import logging
import hashlib


class YabsAwapsMediascopeExport(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        mssql_server = sdk2.parameters.String("MSSQL BannerStorage server:port", default="rapidc:1433", required=True)
        mssql_database = sdk2.parameters.String("MSSQL BannerStorage database", default="BannerStorage", required=True)
        mssql_user = sdk2.parameters.String("MSSQL BannerStorage user", default="yt_loader", required=True)
        mssql_password_vault_name = sdk2.parameters.String("MSSQL BannerStorage user password (vault name)", default="yt_loader_rapidc_password", required=True)

        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn")
        yt_table_path = sdk2.parameters.String("YT table path", default="//home/yabs-rt/ads-verify/mediascope/display/campaigns/")
        mds_host = sdk2.parameters.String("MDS host", default="https://s3.mds.yandex.net")
        mds_bucket_name = sdk2.parameters.String("MDS bucket name", default="mediascope-bucket")
        mds_object_key = sdk2.parameters.String("MDS object key", default="DisplayCampaigns_")

        # use the same secret as tns log uploader BSDEV-76107
        yav_secret = sdk2.parameters.YavSecret("yav secret id",
                                               required=True,
                                               description='secret should contain keys: '
                                                           's3_mds_access_secret_key '
                                                           's3_mds_access_key_id '
                                                           'yt_token')

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("pymssql", version="2.1.4"),
            environments.PipEnvironment('sqlalchemy', version="1.3.23"),
        )
        dns = DnsType.DNS64  # for external interactions

    def _connect_to_mssql(self):
        from sqlalchemy import create_engine
        from sqlalchemy.orm import sessionmaker
        from sqlalchemy.pool import NullPool

        host = self.Parameters.mssql_server
        database = self.Parameters.mssql_database
        user = self.Parameters.mssql_user
        password = sdk2.task.Vault.data(self.author, self.Parameters.mssql_password_vault_name)

        logging.info("Connecting to mssql server {host}".format(host=host))
        engine = create_engine('mssql+pymssql://{username}:{password}@{host}/{database}'.format(host=host, database=database, username=user, password=password),
                               poolclass=NullPool)
        session = sessionmaker(bind=engine)
        logging.info("Successfully connected")

        return session()

    def _connect_to_yt(self):
        import yt.wrapper as yt

        yt_cluster = self.Parameters.yt_cluster
        yt_token = self.Parameters.yav_secret.data()["yt_token"]

        logging.info("Creating YT client for {cluster}".format(cluster=yt_cluster))
        yt_client = yt.YtClient(yt_cluster, token=yt_token)
        logging.info("Successfully created")

        return yt_client

    def _connect_to_s3(self):
        mds_host = self.Parameters.mds_host
        secret = self.Parameters.yav_secret.data()
        mds_secret_access_key = secret["s3_mds_access_secret_key"]
        mds_access_key_id = secret["s3_mds_access_key_id"]

        logging.info("Connecting to S3 {host}".format(host=mds_host))
        session = boto3.session.Session(
            aws_access_key_id=mds_access_key_id,
            aws_secret_access_key=mds_secret_access_key,
        )
        s3_client = session.client(
            service_name='s3',
            endpoint_url=mds_host,
        )
        logging.info("Successfully connected")

        return s3_client

    def load_mediascope_banners(self):
        session = self._connect_to_mssql()
        try:
            logging.info("Loading mediascope banners data...")

            result = session.execute(
                """
                if object_id('tempdb..#msidbanner') is not null drop table #msidbanner

                ;with param_msid as (
                    select
                        c.nmb as bs_creative_nmb,
                        s.value as msid
                    from t_creative c with(nolock)
                        join t_creative_version cv on cv.creative_nmb = c.nmb and cv.is_last_version = 1
                        join t_creative_version_param_value pv on pv.creative_version_nmb = cv.nmb and pv.parameter_nmb = 608 -- MEDIASCOPE_PIXEL_MSID
                        join t_string s on s.nmb = pv.string_nmb
                )
                , param_file as (
                    select
                        c.nmb as bs_creative_nmb,
                        f.stillage_file_url as file_url
                    from t_creative c with(nolock)
                        join t_creative_version cv on cv.creative_nmb = c.nmb and cv.is_last_version = 1
                        join t_creative_version_param_value pv on pv.creative_version_nmb = cv.nmb and pv.parameter_nmb = 313  -- VIDEO
                        join t_file_instance fi on fi.nmb = pv.file_nmb
                        join (select nmb, stillage_file_url from  t_file) f on f.nmb = fi.file_nmb
                )
                select msid,
                        m.bs_creative_nmb as bs_creative_nmb ,
                        file_url
                into #msidbanner
                    from param_msid m with(nolock)
                        join param_file f on m.bs_creative_nmb = f.bs_creative_nmb
                order by bs_creative_nmb

                select msid,
                        link.outer_object_id as cid,
                        m.bs_creative_nmb as bid,
                        file_url as creative_url,
                        ado_cm.name campaign_name,
                        cast(ado_ln.date_begin as date) start_dt,
                        cast(ado_ln.date_end as date) finish_dt
                    from #msidbanner as m with(nolock)
                        join prontoc.ado_current.dbo.vt_creative ado_cr on ado_cr.moderation_identity = m.bs_creative_nmb
                        join prontoc.ado_current.dbo.vr_creative_schedule ado_cs on ado_cs.creative_nmb = ado_cr.nmb
                        join prontoc.ado_current.dbo.vt_schedule ado_sh on ado_sh.nmb = ado_cs.schedule_nmb
                        join prontoc.ado_current.dbo.vt_line_date ado_ln on ado_ln.nmb = ado_sh.line_nmb
                        join prontoc.ado_current.dbo.vt_campaign ado_cm on ado_cm.nmb = ado_ln.campaign_nmb
                        join prontoc.ADO_Integration.dbo.vt_Awaps_Object_Link link on link.object_id = ado_sh.id and link.table_id = 'EAFA2391-2B7B-47D1-A474-8509E1CE780A'
                """
            )
            banners = [dict(r) for r in result]

            logging.info("{} rows loaded".format(len(banners)))

            return banners
        finally:
            session.close()

    def write_banners_to_yt(self, table_path_prefix, banners):
        import yt.wrapper as yt

        def cast_to_yt(value):
            if isinstance(value, date):
                return value.strftime('%Y-%m-%d')
            elif isinstance(value, datetime):
                return value.strftime('%Y-%m-%dT%H:%M:%S')
            return value

        yt_client = self._connect_to_yt()

        table_path = table_path_prefix + datetime.now().strftime("%Y-%m-%d")
        if not yt_client.exists(table_path):
            logging.info("Creating YT table {table}".format(table=table_path))
            yt_client.create('table', table_path, recursive=False, attributes={
                'schema': [
                    {'name': 'msid', 'type': 'string', 'required': True, 'sort_order': 'ascending'},
                    {'name': 'cid', 'type': 'uint32', 'required': True, 'sort_order': 'ascending'},
                    {'name': 'bid', 'type': 'uint32', 'required': True, 'sort_order': 'ascending'},
                    {'name': 'creative_url', 'type': 'string'},
                    {'name': 'campaign_name', 'type': 'string'},
                    {'name': 'start_dt', 'type': 'string'},
                    {'name': 'finish_dt', 'type': 'string'}
                ]
            })
            logging.info("Table created")

        yt_data = sorted(
            [{k: cast_to_yt(v) for k, v in b.items()} for b in banners],
            key=lambda k: (k["msid"], k["cid"], k["bid"])
        )
        logging.info("Writing data to YT...")
        yt_client.write_table(table_path, yt_data, format=yt.JsonFormat(attributes={"encode_utf8": False}, encoding='utf-8'))
        logging.info("{} rows written".format(len(yt_data)))

    def write_banners_to_s3(self, bucket_name, object_key_prefix, banners):
        s3_client = self._connect_to_s3()

        s3_banners = [{
            "position_id": "{msid}_5-{cid}-{bid}".format(msid=b["msid"], cid=b["cid"], bid=b["bid"]),
            "position_name": "{campaign_name}-{bid}".format(campaign_name=b["campaign_name"], bid=b["bid"]),
            "creative_id": "{bid}".format(bid=b["bid"]),
            "creative_url": "{creative_url}".format(creative_url=b["creative_url"]),
            "start_dt": "{start_dt}".format(start_dt=b["start_dt"]),
            "finish_dt": "{finish_dt}".format(finish_dt=b["finish_dt"]),
            "params": [
                {
                    "param_name": "campaign_name",
                    "param_value": "{campaign_name}".format(campaign_name=b["campaign_name"])
                }
            ]
        } for b in banners]

        file_name = object_key_prefix + datetime.now().strftime("%Y-%m-%d")
        object_key = hashlib.sha512(file_name.encode('utf-8')).hexdigest()
        logging.info("Sending {file_name} data as {object_key} to S3...".format(file_name=file_name, object_key=object_key))
        response = s3_client.put_object(Bucket=bucket_name,
                                        Key=object_key,
                                        ContentType='application/json',
                                        Body=json.dumps(s3_banners, ensure_ascii=False))
        logging.info("Response code is {code}".format(code=response['ResponseMetadata']['HTTPStatusCode']))

    def on_execute(self):
        banners = self.load_mediascope_banners()
        if self.Parameters.yt_table_path:
            self.write_banners_to_yt(self.Parameters.yt_table_path, banners)
        if self.Parameters.mds_bucket_name and self.Parameters.mds_object_key:
            self.write_banners_to_s3(self.Parameters.mds_bucket_name, self.Parameters.mds_object_key, banners)
