import logging
import os

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.notification as ctn
from sandbox.sdk2.helpers import subprocess as sp


def get_binary_path(name):
    resource = sdk2.Resource.find(attrs=dict(name=name)).first()
    if resource:
        logging.info('Got resource %i' % resource.id)
        return str(sdk2.ResourceData(resource).path)
    else:
        raise common.errors.TaskError('Binary resource does not exist')


class BananaLinksCollector(sdk2.Resource):
    executable = True


class BananaLinksChecker(sdk2.Resource):
    executable = True


class BananaLinksValidator(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        cluster_name = sdk2.parameters.String(
            label='Cluster name',
            default='hahn',
            required=True,
        )

        collect_links = sdk2.parameters.Bool(
            label='Collect links',
            default=True,
        )

        check_links = sdk2.parameters.Bool(
            label='Check links',
            default=True,
        )

        send_notifications_to = sdk2.parameters.List(
            label='Send notofocations to',
            default=["rarslanova"],
        )

        yt_token_vault_name = sdk2.parameters.String(
            label='YQL token vault name',
            default="YQL_TOKEN",
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def send_invalid_banners_notification(self, table_pathes_list, recipients):
        logging.info('Start sending notification to: {}'.format(recipients))
        body = 'You can check invalid banner info by this links:\n{}'.format("\n".join(table_pathes_list))
        self.server.notification(
            subject='Invalid links detected',
            body=body,
            recipients=recipients,
            transport=ctn.Transport.EMAIL,
        )

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_name)
        cluster_name = self.Parameters.cluster_name
        collect_links = self.Parameters.collect_links
        check_links = self.Parameters.check_links
        send_notifications_to = self.Parameters.send_notifications_to

        os.environ['YT_TOKEN'] = yt_token

        if collect_links:
            logging.info('Start banana-links-collector process')
            binary_path = get_binary_path(name='banana-links-collector-1')
            with sdk2.helpers.ProcessLog(self, "links-collector") as pl:
                subprocess_command = '{} --proxy {}'.format(binary_path, cluster_name)
                p = sp.Popen(subprocess_command, shell=True, stdout=sp.PIPE, stderr=pl.stderr)
                out, _ = p.communicate()
                links_collector_status = p.returncode
                assert not links_collector_status

        invalid_links_table_pathes = []
        if check_links:
            logging.info('Start banana-links-checker process')
            binary_path = get_binary_path(name='banana-links-checker-2')
            for table in ['DirectLinks']:
                with sdk2.helpers.ProcessLog(self, "links-checker-{}".format(table)) as pl:
                    subprocess_command = '{} --proxy {} --links {}'.format(binary_path, cluster_name, table)
                    p = sp.Popen(subprocess_command, shell=True, stdout=sp.PIPE, stderr=pl.stderr)
                    out, _ = p.communicate()
                    links_checker_status = p.returncode
                    assert not links_checker_status

                logging.info('banana-links-checker result: {}'.format(out))
                if out.endswith("Has invalid_links=True"):
                    invalid_links_table_pathes.append('https://yt.yandex-team.ru/{}/#page=navigation&path=//home/yabs/banana/Wrong{}'.format(cluster_name, table))

            if invalid_links_table_pathes and send_notifications_to:
                self.send_invalid_banners_notification(invalid_links_table_pathes, send_notifications_to)

        logging.info('invalid_links_table_pathes: {}'.format(invalid_links_table_pathes))
