from __future__ import print_function
import logging
from copy import deepcopy
from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus


UNRELEASED = "unreleased"


def base_bin_task_parameters(auto_search_default=True, release_version_default=UNRELEASED, resource_attrs_default=None):
    class _BaseBinTaskParameters(sdk2.Parameters):
        with sdk2.parameters.Group("Version and task resource") as version_and_task_resource:
            auto_search = sdk2.parameters.Bool(
                "Search for tasks resource automatically",
                default=auto_search_default,
            )
            with auto_search.value[True]:
                release_version = sdk2.parameters.String(
                    "Release version",
                    choices=[(v, v) for _, v in ReleaseStatus.iteritems()] + [(UNRELEASED, UNRELEASED)],
                    default=release_version_default,
                )
                resource_attrs = sdk2.parameters.Dict(
                    "Filter resource by",
                    description="Will be passed to 'attrs' search parameter",
                    default=resource_attrs_default,
                )

    return _BaseBinTaskParameters()


class BaseBinTaskParameters(sdk2.Parameters):
    _base_bin_task_parameters = base_bin_task_parameters()


class BaseBinTaskMixin:
    """
    Mixin class for support binary task execution. Override method on_save.

    Parameters of the task being mixed should either inherit BaseBinTaskParameters or include it.
    """
    def find_and_apply_tasks_binary(self):
        search_params = {
            "state": State.READY,
            "order": "-id",
            "attrs": self.Parameters.resource_attrs,
        }
        available_resource_ids = []
        if self.Parameters.release_version != UNRELEASED:
            for release_type_name, release_type_value in ReleaseStatus.iteritems():
                found_resource = self._find_exact_resource(release_type_value, search_params)
                if found_resource is not None:
                    available_resource_ids.append(found_resource.id)
                if release_type_name == self.Parameters.release_version:
                    break
        else:
            found_resource = self._find_exact_resource(UNRELEASED, search_params)
            if found_resource is not None:
                available_resource_ids.append(found_resource.id)

        if len(available_resource_ids) > 0:
            resource_id = max(available_resource_ids)
            logging.info("Found %d resource", resource_id)
            self.Requirements.tasks_resource = resource_id

    @staticmethod
    def _find_exact_resource(released, search_params):
        _search_params = deepcopy(search_params)
        if released != UNRELEASED:
            _search_params.setdefault("attrs", {}).update({"released": released.lower()})
        return sdk2.service_resources.SandboxTasksBinary.find(**_search_params).first()

    def on_save(self):
        if not self.Requirements.tasks_resource and self.Parameters.auto_search:
            self.find_and_apply_tasks_binary()


class BaseBinTask(BaseBinTaskMixin, sdk2.Task):
    class Parameters(BaseBinTaskParameters):
        pass
