import base64
import difflib
import json
import re
import zlib
import sys


class JsonPainter:
    def __init__(self, keys_to_delete):
        self.keys_to_delete = keys_to_delete

    def apply_json_paints(self, json):
        if isinstance(json, dict):
            keys = list(json.keys())
            for key in keys:
                if key in self.keys_to_delete:
                    del json[key]
                else:
                    self.apply_json_paints(json[key])
        elif isinstance(json, list):
            for value in json:
                self.apply_json_paints(value)
        return json


class HeaderFilter:
    allowed_operations = {'==', '!=', '>', '>=', '<', '<=', 'find', 'not-find'}

    def __init__(self, filter):
        filter = filter.split(' ', 2)
        assert len(filter) == 3, "Wrong filter - '{}'! Expected {{header}} {{operation}} {{value}}".format(' '.join(filter))
        self.key, self.operation, self.value = filter
        if self.value.isdigit():
            self.value = int(self.value)
        assert self.operation in HeaderFilter.allowed_operations


class BaseResponseDiffer(object):
    def __init__(
            self,
            headers_to_replace,
            body_substitutes,
            json_keys_to_delete,
            base64_prefixes,
    ):
        self.headers_to_replace = headers_to_replace
        self.body_substitutes = body_substitutes
        self.json_painter = JsonPainter(json_keys_to_delete)
        self.base64_prefixes = base64_prefixes

        self.is_pre_gzip = False
        self.is_test_gzip = False

    @staticmethod
    def _split_response(response):
        splitted_response = response.split('\r\n\r\n', 1)
        assert len(splitted_response) == 2, 'Incorrect response: cannot cut off the body'
        body = splitted_response[-1]
        if "Transfer-Encoding" in splitted_response[0]:
            body = '\r\n'.join(body.split('\r\n')[1:-3])

        splitted_response = splitted_response[0].split('\r\n')
        status = splitted_response[0].upper()
        headers = splitted_response[1:]
        return status, headers, body

    @staticmethod
    def _json2str(d):
        return json.dumps(d, sort_keys=True, indent=4)

    @staticmethod
    def _headers_to_dict(headers):
        result = dict()

        for header in headers:
            splitted_header = header.split(': ', 1)
            assert len(splitted_header) == 2, 'Incorrect response: cannot parse header {header}'.format(
                header=header
            )
            splitted_header = [x.strip('"') for x in splitted_header]
            result[splitted_header[0]] = splitted_header[1]
        return result

    @staticmethod
    def _string_lists_diff(list_pre, list_test):
        return '\n'.join(difflib.unified_diff(list_pre, list_test, fromfile='pre', tofile='test', lineterm=''))

    @staticmethod
    def _status_diff(status_pre, status_test):
        return str(BaseResponseDiffer._string_lists_diff([status_pre], [status_test]))

    @staticmethod
    def _dict_diff(dict_pre, dict_test):
        serialized_pre = BaseResponseDiffer._json2str(dict_pre).splitlines()
        serialized_test = BaseResponseDiffer._json2str(dict_test).splitlines()
        return BaseResponseDiffer._string_lists_diff(serialized_pre, serialized_test)

    @staticmethod
    def _xml_diff(xml_pre, xml_test):
        serialized_pre = xml_pre.toprettyxml(indent=' ').splitlines()
        serialized_test = xml_test.toprettyxml(indent=' ').splitlines()
        return BaseResponseDiffer._string_lists_diff(serialized_pre, serialized_test)

    @staticmethod
    def try_to_json(s):
        is_json = False
        try:
            parse_s = json.loads(s)
            is_json = isinstance(parse_s, dict)
            if is_json:  # for double quoting (e.g. s = '"{\\"key1\\": 1}"'), parse_s will be unicode string, not dict!
                s = parse_s
        except ValueError:
            pass
        return s, is_json

    def _try_get_json_inside(self, s, quoting_word=None):
        def find_first_json(s, shift=0):
            l = s.find('{', shift)
            if l == -1:
                return -1, -1
            cnt = 1
            i = l + 1
            while i < len(s) and cnt > 0:
                if s[i] == '{':
                    cnt += 1
                elif s[i] == '}':
                    cnt -= 1
                i += 1
            r = i if cnt == 0 else -1
            return l, r

        res = []
        left, right = find_first_json(s)
        while left != -1 and right != -1:
            json_str = s[left:right]
            if quoting_word is not None:
                match = lambda m: m.group(1) + '"{w}":'.format(w=quoting_word)
                substr = '({[^"\\w\']*?)' + quoting_word + ':'
                json_str = re.sub(substr, match, json_str, flags=re.S)  # { AUCTION_DC_PARAMS: {json}} -> { "AUCTION_DC_PARAMS": {json}}
            s_json, is_json = self.try_to_json(json_str)
            if is_json and s_json != {}:
                self.json_painter.apply_json_paints(s_json)
                res += [s[:left], s_json] if left > 0 else [s_json]
                s = s[right:]
                shift = 0
            else:
                shift = left + 1
            left, right = find_first_json(s, shift)
        if len(res) > 0:
            if s != '':
                res.append(s)
            return res
        return s

    def _headers_diff(self, headers_pre, headers_post):
        parsed_headers_pre = self._headers_to_dict(headers_pre)
        parsed_headers_pre.update(self.headers_to_replace)

        parsed_headers_post = self._headers_to_dict(headers_post)
        parsed_headers_post.update(self.headers_to_replace)

        self.is_pre_gzip = 'gzip' in parsed_headers_pre.get('Content-Encoding', '')
        self.is_test_gzip = 'gzip' in parsed_headers_post.get('Content-Encoding', '')

        return str(self._dict_diff(parsed_headers_pre, parsed_headers_post))

    def _decompress(self, entity):
        try:
            entity = zlib.decompress(entity, zlib.MAX_WBITS | 16)
        except:
            try:
                entity = zlib.decompress(entity, 29)
            except:
                pass
        return entity

    @staticmethod
    def apply_substituttes(entity, substitutes):
        for substitute in substitutes.items():
            entity = re.sub(str(substitute[0]), str(substitute[1]), str(entity))
        return entity

    @staticmethod
    def skip_by_headers_filters(headers, filters):  # filters: List[HeaderFilter]
        if not filters:
            return False
        parse_headers = BaseResponseDiffer._headers_to_dict(headers)
        for filter in filters:
            if filter.key not in parse_headers:
                return True
            real_value = parse_headers[filter.key]
            if filter.operation.endswith('find'):
                condition = real_value.find(filter.value) != -1
                if filter.operation == 'not-find':
                    condition = not condition
            elif filter.operation == '==':
                condition = real_value == filter.value
            elif filter.operation == '!=':
                condition = real_value != filter.value
            elif filter.operation == '>':
                condition = real_value > filter.value
            elif filter.operation == '>=':
                condition = real_value >= filter.value
            elif filter.operation == '<':
                condition = real_value < filter.value
            elif filter.operation == '<=':
                condition = real_value <= filter.value
            else:
                raise ValueError('Unknown operation: {}! Expected one of {}'.format(filter.operation, HeaderFilter.allowed_operations))

            if not condition:
                return True
        return False

    @staticmethod
    def remove_single_xml_tags(s, tags):
        for tag in tags:
            s = re.sub('<' + tag + '.*?>', '', s)
        return s

    @staticmethod
    def base64decode(body):
        try:
            decode_body = base64.b64decode(body)
            if base64.b64encode(decode_body) == body:
                return decode_body, True
        except:
            pass
        return body, False

    def _replace_b64decoded(self, body, prefix, request_id=None):
        reg_exp = prefix + r'''(\\?)("|u')([a-zA-Z0-9-=_+\/]+)(\\?)("|')'''
        sub_function = (
            lambda match: (prefix
                           + match.group(1)
                           + match.group(2)
                           + json.dumps(base64.urlsafe_b64decode(match.group(3)))[1:-1]
                           + match.group(4)
                           + match.group(5))
        )
        try:
            if request_id is not None and re.search(reg_exp, body) is not None:
                print >>sys.stderr, self.request_id, prefix
            return re.sub(reg_exp, sub_function, body)
        except Exception, e:
            if request_id is not None:
                print >>sys.stderr, "_replace_b64decoded error:", request_id, str(e)
            return body

    def _replace_all_base64(self, entity, base64_prefixes, request_id=None):
        for prefix in base64_prefixes:
            entity = self._replace_b64decoded(str(entity), str(prefix), request_id)
        return entity

    def diff_response_data(self, response_pre, response_test):
        status_pre, headers_pre, body_pre = self._split_response(response_pre)
        status_test, headers_test, body_test = self._split_response(response_test)

        status_diff = self._status_diff(status_pre, status_test)
        headers_diff = self._headers_diff(headers_pre, headers_test)
        return status_diff, headers_diff, body_pre, body_test
