import os
import json
import logging

from sandbox import common
from sandbox.common.utils import chain
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Semaphores


from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.common.utils import get_or_default

from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.common.yabs.server.db.task.basegen import FETCH_ID_KEY, FETCH_OUTPUTS_KEY, CS_IMPORT_ID_KEY
from sandbox.projects.common.yabs.server.db.task.cs import YtPool
from sandbox.projects.common.yabs.server.util.general import check_tasks

from sandbox.projects.yabs.bases.base_producing_task import BinBasesProducingTask
from sandbox.projects.yabs.bases.keys import UNPACKED_SIZE_ATTR_KEY, CHKDB_RESOURCE_ATTR_KEY, CHKDB_WITHOUT_HASH_RESOURCE_ATTR_KEY
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_BASE_CHKDB, YABS_SERVER_BASE_CHKDB_WITHOUT_HASH
from sandbox.projects.yabs.qa.tasks.YabsServerRealRunCSImport import IMPORT_PREFIX_KEY

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls


class RInput(SandboxStringParameter):
    name = 'rinput'
    description = 'Input tables  for reduce (--rinput) (not used with fetch)'
    multiline = True
    default_value = ''


class YabsServerBasesReduce(BinBasesProducingTask):

    type = 'YABS_SERVER_BASES_REDUCE'

    required_ram = 20 * 1024

    execution_space = 100 * 1024

    client_tags = Tag.GENERIC & Tag.LINUX_PRECISE

    input_parameters = (RInput, ) + BinBasesProducingTask.input_parameters

    environment = (PipEnvironment('yandex-yt', use_wheel=True), ) + BinBasesProducingTask.environment

    privileged = False  # Check parents for privileged!

    def initCtx(self):
        ctx = super(YabsServerBasesReduce, self).initCtx()
        ctx.update(kill_timeout=5 * 3600)
        return ctx

    def setup_semaphores(self):
        yt_pool = get_or_default(self.ctx, YtPool)
        if not yt_pool or yt_pool == yt_bases.YT_POOL:
            yt_pool_semaphore = self.YT_POOL_REDUCE_SEMAPHORE
        else:
            yt_pool_semaphore = yt_pool
        self.semaphores(Semaphores(
            acquires=[
                Semaphores.Acquire(name=yt_pool_semaphore, weight=3)
            ]
        ))

    @trace_calls
    def generate_bases(self, dbs_to_generate, target_dir, db_ver):

        if any(_.startswith('dblm') for _ in dbs_to_generate):
            raise common.errors.TaskFailure("This task cannot generate dblm* bases")

        yt_token = self.get_yt_token()  # Fail fast if we have no token available

        self._wait_basegen_prepare_tasks()

        self.report_operations()
        base_paths, check_data = self._generate_yabscs(yt_token, dbs_to_generate, target_dir, db_ver)

        for base_tag, path in base_paths.iteritems():
            os.rename(os.path.abspath(path), os.path.abspath(self.get_packed_base_name(db_ver, base_tag)))
            res_id = dbs_to_generate[base_tag]
            try:
                chkdb_base = check_data[base_tag]
                size = chkdb_base.pop('bases.check_size')[0]
                chkdb_dump = chkdb_base.pop('bases.dump')[0]
            except KeyError:
                raise common.errors.TaskFailure("{tag}, bases.dump or bases.check_size for {tag} not found in stored chkdb output".format(tag=base_tag))
            chkdb_path = 'chkdb_%s' % base_tag
            chkdb_path_without_hash = chkdb_path + '_without_hash'
            with open(chkdb_path_without_hash, 'w') as f:
                json.dump(chkdb_base, f, sort_keys=True, indent=0)
            with open(chkdb_path, 'w') as f:
                f.write(chkdb_dump)
            chkdb_without_hash_res_id = self.create_resource(
                description="yabs_chkdb of %s" % base_tag,
                resource_type=YABS_SERVER_BASE_CHKDB_WITHOUT_HASH,
                resource_path=chkdb_path_without_hash,
                attributes={'base_name': base_tag}
            ).id
            chkdb_res_id = self.create_resource(
                description="yabs_chkdb of %s" % base_tag,
                resource_type=YABS_SERVER_BASE_CHKDB,
                resource_path=chkdb_path,
                attributes={'base_name': base_tag}
            ).id
            for attr_name, attr_value in [(CHKDB_WITHOUT_HASH_RESOURCE_ATTR_KEY, chkdb_without_hash_res_id), (CHKDB_RESOURCE_ATTR_KEY, chkdb_res_id), (UNPACKED_SIZE_ATTR_KEY, size)]:
                self.set_or_update_resource_attribute(res_id, attr_name, attr_value)

    @trace_calls
    def _wait_basegen_prepare_tasks(self):
        basegen_prepare_ids = list(chain(
            self.ctx.get(FETCH_ID_KEY, []) or [],
            self.ctx.get(CS_IMPORT_ID_KEY, []) or [],
        ))
        if not basegen_prepare_ids:
            return []

        check_tasks(self, basegen_prepare_ids)
        return basegen_prepare_ids

    def _get_fetch_outputs(self, rest_client):
        fetch_outputs = []
        for fetch_id in self.ctx.get(FETCH_ID_KEY, []):
            ctx = rest_client.task[fetch_id].context.read()
            fetch_outputs += ctx[FETCH_OUTPUTS_KEY]
        return fetch_outputs

    def _get_import_prefix(self, rest_client):
        if self.import_prefix:
            logging.info("Get prefix for import from input parameter:", self.import_prefix)
            return self.import_prefix
        cs_import_id = self.ctx.get(CS_IMPORT_ID_KEY)
        if cs_import_id:
            subtask_context = rest_client.task[cs_import_id].context.read()
            import_prefix = subtask_context.get(IMPORT_PREFIX_KEY)
            logging.info("Get prefix for import from child task:", import_prefix)
            return import_prefix
        raise common.errors.TaskFailure("There is no source of import_prefix (input parameter or cs_import_id)")

    @trace_calls
    @yt_bases.yabscs_failures_retried
    def _generate_yabscs(self, yt_token, dbs_to_generate, target_dir, db_ver):
        input_spec_res_id = self.input_spec_res_id
        input_spec_path = self.sync_resource(input_spec_res_id) if input_spec_res_id else None

        rest_client = common.rest.Client()
        rinput = get_or_default(self.ctx, RInput).splitlines() + self._get_fetch_outputs(rest_client)
        import_prefix = self._get_import_prefix(rest_client)

        return yt_bases.reduce(
            yt_token=yt_token,
            bs_release_yt_dir=self.get_yabscs(),
            base_tags=dbs_to_generate.viewkeys(),
            sandbox_client_fqdn=self.client_info['fqdn'],
            date=self.get_db_date(),
            target_dir=target_dir,
            expected_base_ver=db_ver,
            import_prefix=import_prefix,
            input_spec_path=input_spec_path,
            rinput=rinput,
            settings_spec=self.cs_settings,
            task_id=self.id,
            yt_pool=get_or_default(self.ctx, YtPool),
        )
