import datetime
import itertools
import json
import logging
import re
import time
from collections import defaultdict

import sandbox.projects.common.yabs.server.db.utils as dbutils
from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.yabs.qa.utils.base import truncate_base_prefix
from sandbox.projects.yabs.qa.utils.general import (
    calc_md5_from_json,
    get_json_md5
)
from sandbox.projects.yabs.qa.utils.importer import get_bases_importers_with_dependencies


logger = logging.getLogger(__name__)


def make_child_descr(db_ver, tags, parent_descr):
    '''
    >>> make_child_descr(1234567890, ['dbe', 'banner_update_01', 'st001', 'st012', 'st123', 'resource01_1', 'resource99_6'], 'ZXCVBN')
    '1234567890 | banner_update_01 dbe resource{01_1, 99_6} st{001, 012, 123} | ZXCVBN'
    '''
    groups = defaultdict(list)
    for tag in tags:
        m = re.match(r'^(\w+?)(\d+|\d+_\d+)$', tag)
        if m is None:
            groups[tag].append('')
        else:
            groups[m.group(1)].append(m.group(2))

    return '{} | {} | {}'.format(
        db_ver,
        ' '.join(k + ('{%s}' % ', '.join(groups[k]) if len(groups[k]) > 1 else groups[k][0]) for k in sorted(groups)),
        parent_descr,
    )


def intersect_dicts(*dicts):
    if not dicts:
        return {}

    return dict(set(dicts[0].items()).intersection(*[d.items() for d in dicts[1:]]))


def serialize_dict(dict_, splitter='_'):
    return splitter.join(
        "{key}={value}".format(key=key, value=dict_[key])
        for key
        in sorted(dict_.keys())
    )


def base_to_importer_settings_version(cs_dir, bases, settings_spec=None):
    """Calculate the checksum of settings for importers involved in generating base

    :param cs_dir: path to CS directory
    :type cs_dir: str
    :param settings_spec: CS settings, defaults to None
    :type settings_spec: dict, optional
    :return: Mapping from base name to md5 hash of importers settings
    :rtype: dict
    """
    settings_version = yt_bases.get_cs_settings_version(cs_dir, settings_spec)
    logger.debug('Importer settings version: %s', settings_version)

    importers_info = yt_bases.get_cs_import_info(cs_dir)
    mkdb_info = dbutils.get_full_mkdb_info(cs_dir)
    base_to_importer_settings_checksum = {}
    for base_name in bases:
        importers = get_bases_importers_with_dependencies({base_name}, importers_info, mkdb_info)
        logger.debug('Base "%s" depends on importers %s', base_name, importers)
        importer_versions = {
            importer: settings_version[importer]
            for importer in importers
        }
        base_to_importer_settings_checksum[base_name] = calc_md5_from_json(json.dumps(importer_versions))

    return base_to_importer_settings_checksum


def base_to_importer_code_version(cs_dir, bases):
    """Calculate the checksum of code for importers involved in generating base

    :param cs_dir: path to CS directory
    :type cs_dir: str
    :return: Mapping from base name to md5 hash of importers code
    :rtype: dict
    """
    importer_code_version = dbutils.get_importer_code_ver(cs_dir)
    logger.debug('Importer code version: %s', importer_code_version)

    importers_info = yt_bases.get_cs_import_info(cs_dir)
    mkdb_info = dbutils.get_full_mkdb_info(cs_dir)
    base_to_importer_code_checksum = {}
    for base_name in bases:
        importers = get_bases_importers_with_dependencies({base_name}, importers_info, mkdb_info)
        logger.debug('Base "%s" depends on importers %s', base_name, importers)
        importer_versions = {
            importer: importer_code_version[importer]
            for importer in importers
        }
        base_to_importer_code_checksum[base_name] = calc_md5_from_json(json.dumps(importer_versions))

    return base_to_importer_code_checksum


def base_mkdb_info_version(mkdb_info):
    """Calculate checksum of mkdb_info of all bases

    :param mkdb_info: info for every base, output of 'dbtool mkdb_info'
    :type mkdb_info: dict
    :return: Mapping from base name to md5 hash of mkdb_info
    :rtype: dict
    """
    return {
        base: get_json_md5(mkdb_info_value)
        for base, mkdb_info_value in mkdb_info.items()
    }


def filter_mkdb_info_version_by_bases(mkdb_info_version, bases):
    """Filter mkdb_info_version dict of md5 hashes by base names

    :return: Mapping from short base name to md5 hash of mkdb_info
    :rtype: dict
    """
    short_base_names = {
        truncate_base_prefix(base)
        for base in bases
    }
    return {
        base: ver
        for base, ver in mkdb_info_version.items()
        if base in short_base_names
    }


def lock_for_search(yt_client, lock_path, attributes):
    from yt.common import YtError

    lock_child_key = get_json_md5(attributes)
    logger.debug('Lock\'s child_key: %s', lock_child_key)

    lock_timeout = datetime.timedelta(minutes=15).total_seconds() * 1000
    logger.debug(
        'Try to acquire waitable (%d ms) shared lock on node "%s" with child_key="%s"',
        lock_timeout, lock_path, lock_child_key)

    tic = time.time()

    try:
        yt_client.lock(
            lock_path,
            mode='shared',
            child_key=lock_child_key,
            waitable=True,
            wait_for=lock_timeout,
        )
    except YtError as e:
        logger.warning('Failed to acquire lock\n%s', e)
    else:
        lock_waiting_time = (time.time() - tic) * 1000
        logger.debug('Lock has been acquired, waited for %f milliseconds', lock_waiting_time)


def get_importer_queries(importers_info):
    return list(set(itertools.chain.from_iterable(data.get("mkdb_queries", []) for data in importers_info.values())))


def get_mysql_queries(mkdb_info):
    return [
        query["Name"]
        for query in mkdb_info["Queries"]
        if query["Type"] == "MySql"
    ]


def base_requires_mysql(mkdb_info, importer_queries):
    mysql_queries = get_mysql_queries(mkdb_info)
    if not mysql_queries:
        return False

    return not set(mysql_queries).issubset(importer_queries)
