# -*- coding: utf-8 -*-
import json
import logging
import os

import sandbox
import sandbox.projects.common.yabs.server.db.utils as dbutils
from sandbox import sdk2
from sandbox.projects.yabs.qa import resource_types
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.common.yabs.server.db.task.basegen import (
    BinBasesTask,
    BIN_BASE_RES_IDS_KEY,
    GENERATION_ID_KEY,
    BaseTagImporterSettingsVersion,
    BaseTagImporterCodeVersion,
    BaseTagMkdbInfoVersion,
)
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_entry_point


logger = logging.getLogger()


class OutputResourceTTL(sdk2.parameters.Integer):
    name = 'output_resource_ttl'
    description = 'TTL for binary bases'
    default = resource_types.YABS_SERVER_B2B_BINARY_BASE.ttl
    default_value = resource_types.YABS_SERVER_B2B_BINARY_BASE.ttl


class DBVer(SandboxStringParameter):
    name = 'db_ver'
    description = 'Base version for resource creation'
    required = False


class DBInternalVers(SandboxStringParameter):
    name = 'db_internal_vers'
    description = 'Internal base versions for resource creation'
    required = False
    default_value = '{}'


class BinBasesProducingTask(BinBasesTask):  # pylint: disable=R0904
    """Common base for YabsServerMakeBinBases subtasks that have binary bases as output resources"""
    max_restarts = 4

    input_parameters = (DBVer, DBInternalVers, OutputResourceTTL, ) + BinBasesTask.input_parameters

    def setup_semaphores(self):
        raise NotImplementedError("Class is abstract")

    def generate_bases(self, dbs_to_generate, initial_work_dir, db_ver):
        raise NotImplementedError("Class is abstract")

    @staticmethod
    def set_or_update_resource_attribute(db_res_id, attr_name, attr_value):
        rest_client = sandbox.common.rest.Client()
        data = {'name': attr_name, 'value': str(attr_value)}
        try:
            rest_client.resource[db_res_id].attribute.create(data)
        except Exception:  # FIXME use appropriate classes of exceptions?
            logging.warning("Failed to create attribute %s: %s for resource %s. Will try update.", attr_name, attr_value, db_res_id, exc_info=True)
            rest_client.resource[db_res_id].attribute[attr_name].update(data)

    def on_enqueue(self):
        self.ctx['kill_timeout'] = 5 * 3600
        self.setup_semaphores()

        # Create output resources in advance for better binary bases reusing
        db_ver = get_or_default(self.ctx, DBVer)
        if db_ver:
            db_internal_vers = json.loads(get_or_default(self.ctx, DBInternalVers))
            ttl = get_or_default(self.ctx, OutputResourceTTL)
            self._create_output_resources(db_ver, db_internal_vers, ttl=ttl)

    def _create_output_resources(self, db_ver, db_internal_vers, ttl=resource_types.YABS_SERVER_B2B_BINARY_BASE.ttl):
        """Can be executed in on_enqueue"""
        res_ids = []
        binary_bases_attrs = self.iter_binary_bases_attrs(
            tags=self.get_db_list(),
            db_ver=db_ver,
            db_internal_vers=db_internal_vers,
            base_settings_version=json.loads(get_or_default(self.ctx, BaseTagImporterSettingsVersion)),
            base_code_version=json.loads(get_or_default(self.ctx, BaseTagImporterCodeVersion)),
            base_mkdb_info_version=json.loads(get_or_default(self.ctx, BaseTagMkdbInfoVersion)),
        )
        for attrs in binary_bases_attrs:
            base_tag = attrs['tag']
            attrs.update({
                'import_digest': self.import_digest,
                'import_prefix': self.import_prefix,
                'input_spec': str(self.input_spec_res_id),
                'cs_import_ver': str(self.cs_import_ver),
                'common_oneshots_md5': self.common_oneshots_md5,
                'sync_upload_to_mds': True,
                'ttl': ttl,
            })

            res = self.create_resource(
                description="{}.{}.yabs".format(db_ver, base_tag),
                resource_path=self.get_packed_base_name(db_ver, base_tag),
                resource_type=resource_types.YABS_SERVER_B2B_BINARY_BASE,
                attributes=attrs
            )
            res_ids.append(res.id)

        self.ctx[BIN_BASE_RES_IDS_KEY] = res_ids

    def get_generation_id(self):
        return self.ctx.get(GENERATION_ID_KEY, 0)

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        params_db_ver = get_or_default(self.ctx, DBVer)
        params_db_internal_vers = json.loads(get_or_default(self.ctx, DBInternalVers))
        tooldir = self.get_yabscs()
        db_ver = dbutils.get_base_ver(tooldir)
        db_internal_vers = dbutils.get_base_internal_vers(tooldir)
        if params_db_ver and params_db_ver != db_ver:
            raise RuntimeError("Base version from task input parameter (%s) and dbtool (%s) are different" % (params_db_ver, db_ver))

        if not (params_db_ver and params_db_internal_vers):
            self._create_output_resources(db_ver, db_internal_vers)

        dbs_to_generate = self._get_not_ready_resources()

        self.generate_bases(dbs_to_generate, os.getcwd(), db_ver)

    def _get_not_ready_resources(self):
        rest_client = sandbox.common.rest.Client()
        result = dict()
        for res_id in self.ctx[BIN_BASE_RES_IDS_KEY]:
            res = rest_client.resource[res_id].read()
            if res['state'] != 'READY':
                db_type = res['attributes']['tag']
                result[db_type] = res_id
        return result
