import os
import json
import requests
import logging

import yt.wrapper as yt


def adapter(yt_cluster, yt_token, bigb_prefix, bstr_prefix, versions_count=1, multi_versions_mode=False):
    yt.config["proxy"]["url"] = yt_cluster
    yt.config["token"] = yt_token

    # process packs for [count] recent versions
    versions = yt.list(yt.ypath_join(bigb_prefix, "links"))
    recent_versions = sorted(versions, reverse=True)[0:versions_count if multi_versions_mode else 1]

    if multi_versions_mode:  # cleanup old version directories
        bstr_versions = yt.list(yt.ypath_join(bstr_prefix))
        for version in bstr_versions:
            if version not in recent_versions:
                path = yt.ypath_join(bstr_prefix, version)
                logging.info("removing old version directory {}".format(path))
                yt.remove(path, recursive=True)

    # processing packs
    for version in recent_versions:
        logging.info("processing version {}".format(version))

        if multi_versions_mode:
            bstr_path = yt.ypath_join(bstr_prefix, version)
        else:
            bstr_path = bstr_prefix

        # get last processed node id
        node_id_path = yt.ypath_join(bstr_path, "_node_id")
        try:
            last_node_id = yt.get(node_id_path)
        except yt.errors.YtResolveError:
            last_node_id = None
            logging.info("{} does not exists, assume that it is new version".format(node_id_path))

        # get current node id
        path = yt.ypath_join(bigb_prefix, "links", version, "valid")
        try:
            node_id = yt.get_attribute(path, "id")
        except yt.errors.YtResolveError:
            node_id = None
            logging.warning("broken link {}, skip".format(path))
            continue

        if node_id == last_node_id:  # node_id changed, process new pack
            logging.info("node is not changed, skip")
            continue

        pack = yt.get(path)
        common_attributes = {k: v for k, v in pack.items() if k != "data"}  # get pack common attributes

        # for each resource in pack fill bstr_info and create node
        for resource_info in json.loads(pack["data"]):
            resource_info.update(common_attributes)

            # get resource sandbox info
            resource_id = resource_info["source"].split(":")[1]
            sandbox_info = requests.get("https://sandbox.yandex-team.ru/api/v1.0/resource/{}".format(resource_id)).json()

            bstr_info = {
                "fsize": sandbox_info["size"],
                "mtime": resource_info["created"],
                "name": resource_info["alias"],
                "now": resource_info["created"],
                "origin": "bigb-bstr-adapter",
                "priority": 1,
                "torrent": sandbox_info["skynet_id"],
                "extra": resource_info,
            }

            # and create node
            path = yt.ypath_join(bstr_path, resource_info["alias"])
            yt.create(
                path=path,
                type="document",
                recursive=True,
                attributes={"_bstr_info": bstr_info},
                force=True
            )

        # update _node_id for skipping next time if not changed
        last_node_id = yt.set(node_id_path, node_id)


if __name__ == "__main__":
    logging.basicConfig(format='%(levelname)s:%(message)s', level=logging.INFO)

    yt_cluster = "locke"
    with open(os.path.expanduser("~/.yt/token")) as fd:
        yt_token = fd.read().strip()

    adapter(
        yt_cluster=yt_cluster,
        yt_token=yt_token,
        bigb_prefix="//home/bigb/valid_resources/eagle_1",
        bstr_prefix="//home/yabs/tmp/bigb_bstr",
        versions_count=1,
        multi_versions_mode=False,
    )
