from __future__ import print_function
import logging
import json
import requests
import time

from collections import defaultdict, Counter
from datetime import datetime

from sandbox import sdk2, common
from sandbox.sdk2 import parameters
from sandbox.projects.yabs.base_bin_task import BaseBinTask


COMPLETE_STATE = 'complete'
PROCESSING_STATE = 'wait_dropstat'

DROPSTAT_TYPE_COSTCUR = 'costcur'
DROPSTAT_TYPE_EVENTCOST = 'eventcost'

SOLOMON_API_URL = "http://api.solomon.search.yandex.net"


class YabsChargeBack(BaseBinTask):
    '''
    ChargeBack
    '''

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        description = 'ChargeBack'

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = parameters.Dict('Filter resource by', default={'name': 'YabsChargeBack'})

        with parameters.Group('YT parameters') as yt_params:
            cluster = parameters.String(
                'YT cluster',
                required=True,
                default='hahn',
            )

            yabs_cs_chback_secret_id = parameters.YavSecret(
                label='YABS Content System secret id',
                required=True,
                description='secret should contain keys: yt-token',
                default='sec-01fxaess4c3fekpqxz84f9pkwt',
            )

            dropstat_secret_id = parameters.YavSecret(
                label='dropstat token secret id',
                required=True,
                description='secret should contain keys: dropstat_token',
                default='sec-01dc7fm1hk0ehgs2p6mh6ftzap',
            )

        with parameters.Group('Execution parameters') as exec_params:
            caesar_orders_prefix = parameters.String(
                'CaesarOrders table path',
                required=True,
                default='//home/bigb/caesar/stable/Orders',
            )
            tmp_folder_path = parameters.String(
                'Tmp folder for yql',
                required=True,
                default='//home/yabs-cs-preprod/chargeback/tmp',
            )
            order_stat_table_path = parameters.String(
                'OrderStat table path',
                required=True,
                default='//home/yabs/dict/PrebillingOrderStat',
            )
            order_stat_day_table_path = parameters.String(
                'OrderStatDay table path',
                required=True,
                default='//home/yabs/stat/OrderStatDay',
            )
            history_table_path = parameters.String(
                'ChargeBack history table path',
                required=True,
                default='//home/yabs-cs-preprod/chargeback/ChargeBackHistory',
            )
            caesar_dump_path = parameters.String(
                'Path to caesar dump table with raw candidates from Caesar',
                required=True,
                default='//home/yabs-cs-preprod/chargeback/caesar_dumps/history',
            )
            chargeback_day_limit = parameters.Integer(
                'Limit in days for ChargeBack candidates',
                required=True,
                default=30,
            )
            chargeback_limit = parameters.Integer(
                'Money limit for ChargeBack',
                required=True,
                default=1e9,
            )
            last_event_time_delay = parameters.Integer(
                'Delay in hours from GroupLastEventTime before ChargeBack',
                required=True,
                default=24,
            )
            minimum_chargeback_waiting_time = parameters.Integer(
                'Minimum chargeback waiting time in hours',
                required=True,
                default=30,
            )
            chargeback_black_list = parameters.List(
                'OrderID black list for ChargeBack',
                required=True,
                default=[2017288],
            )
            chargeback_client_id_black_list = parameters.List(
                'ClientID black list for ChargeBack',
                required=True,
                default=[1397882, 2214386],
            )
            chargeback_client_id_white_list = parameters.List(
                'ClientID white list for ChargeBack',
                required=True,
                default=[],
            )
            dropstat_retries = parameters.Integer(
                'Amount of dropstat requests retries',
                required=True,
                default=5,
            )
            testing_mode = parameters.Integer(
                'DropStat testing mode. 1 for testing requests to dropstat, 0 for production',
                required=True,
                default=1,
            )

    def check_chargeback_limit(self, candidates):
        whitelist_group_money_left_sum = 0
        group_money_left_sum = 0
        costcur_count = 0
        eventcost_count = 0

        chargeback_client_id_white_list = set(self.Parameters.chargeback_client_id_white_list)

        unique_groups = set()

        for candidate in candidates:
            if candidate['GroupOrderID'] not in unique_groups:
                if candidate['IsCurrencyOrder']:
                    costcur_count += 1
                    group_money_left = int(candidate['GroupMoneyLeftCur'] * candidate['OrderCost'] / (candidate['OrderCostCur'] * 0.82))
                    if str(candidate['ClientID']) in chargeback_client_id_white_list:
                        logging.info('Processing ClientID {} from white list'.format(candidate['ClientID']))

                        whitelist_group_money_left_sum += group_money_left
                    else:
                        group_money_left_sum += group_money_left
                else:
                    eventcost_count += 1
                    if str(candidate['ClientID']) in chargeback_client_id_white_list:
                        logging.info('Processing ClientID {} from whitelist'.format(candidate['ClientID']))

                        whitelist_group_money_left_sum += candidate['GroupMoneyLeft']
                    else:
                        group_money_left_sum += candidate['GroupMoneyLeft']
                unique_groups.add(candidate['GroupOrderID'])

        logging.info('Whitelist sum in chips to be charged back: {}'.format(whitelist_group_money_left_sum))
        logging.info('Total sum in chips to be charged back: {}'.format(group_money_left_sum + whitelist_group_money_left_sum))

        if -group_money_left_sum > self.Parameters.chargeback_limit:
            raise common.errors.TaskError("ChargeBack Limit is exceeded. Check logs")

        self.solomon_client.set_value(
            sensor='cost',
            value=-group_money_left_sum - whitelist_group_money_left_sum,
        )

        self.solomon_client.set_value(
            sensor='orders_count',
            value=costcur_count,
            labels={
                'dropstat_type': DROPSTAT_TYPE_COSTCUR,
                'state': 'new',
            },
        )

        self.solomon_client.set_value(
            sensor='orders_count',
            value=eventcost_count,
            labels={
                'dropstat_type': DROPSTAT_TYPE_EVENTCOST,
                'state': 'new',
            },
        )

    def dropstat_request(self, date, order_ids, limits, limit_by):
        headers = {'Content-Type': 'application/json', 'Authorization': 'OAuth {}'.format(self.dropstat_token)}
        url = 'https://dropstat-yabs.in.yandex-team.ru/create'
        params = {
            'log_type': 'chevent',
            'drop_type': 'partial_multi',
            'limit_by': limit_by,
            'limit_key': 'orderid',
            'limit_map': json.dumps(limits),
            'orderid': ','.join(map(str, order_ids)),
            'sum_estimation': '123',
            'fraud_keyword': 'chargeback',
            'testing': self.Parameters.testing_mode,
            'date_from': date,
            'date_to': date,
            'time_resolution': '1d',
            'without_oneshot': '1',
        }

        try:
            resp = requests.post(url, json=params, headers=headers)
        except Exception as e:
            logging.info('Failed dropstat request: {}'.format(e))
            return False

        try:
            resp.raise_for_status()
        except Exception as e:
            logging.info('Bad request. Response: {}'.format(e))
            return False

        logging.info('DropStat Request:\n{}\n'.format(json.dumps(resp.json(), indent=2)))

        return True

    def safe_dropstat_request(self, dropstat_requests_by_day, limit_by):
        charged_back = Counter()

        for date, limits in dropstat_requests_by_day.items():
            for _ in range(self.Parameters.dropstat_retries):
                is_success = self.dropstat_request(date, limits.keys(), limits, limit_by)
                limits = Counter(limits)
                if is_success:
                    logging.info('Date: {}\nLimitMap:{}\n\n'.format(date, json.dumps(limits, indent=2)))

                    charged_back += limits
                    break

        return charged_back

    def prepare_dropstat_request(self, yt, candidates):
        if not candidates:
            return {}

        cost_by_day = yt.select_rows('''
                OrderID,
                UpdateTime,
                Cost,
                CostCur
            from
                [{order_stat_day}]
            where
                OrderID in ({orders})
                and UpdateTime > {cur_time} - 60 * 60 * 24 * {chargeback_day_limit}
            order by UpdateTime desc
            limit 1000000
        '''.format(
            order_stat_day=self.Parameters.order_stat_day_table_path,
            orders=', '.join([str(row['OrderID']) for row in candidates]),
            cur_time=int(time.time()),
            chargeback_day_limit=self.Parameters.chargeback_day_limit,
        ), input_row_limit=10**7)

        dropstat_type = {}
        order_chargeback = {}
        group_chargeback = {}
        group_id_by_order_id = {}

        for row in candidates:
            dropstat_type[row['OrderID']] = DROPSTAT_TYPE_COSTCUR if row['IsCurrencyOrder'] else DROPSTAT_TYPE_EVENTCOST
            group_id_by_order_id[row['OrderID']] = row['GroupOrderID']

            if dropstat_type[row['OrderID']] == DROPSTAT_TYPE_COSTCUR:
                order_chargeback[row['OrderID']] = row['OrderCostCur']
                group_chargeback[row['GroupOrderID']] = -row['GroupMoneyLeftCur']
            else:
                order_chargeback[row['OrderID']] = row['OrderCost']
                group_chargeback[row['GroupOrderID']] = -row['GroupMoneyLeft']

        dropstat_costcur_requests_by_day = defaultdict(dict)
        dropstat_eventcost_requests_by_day = defaultdict(dict)

        for row in cost_by_day:
            date = datetime.fromtimestamp(row['UpdateTime']).strftime('%Y-%m-%d')
            group_id = group_id_by_order_id[row['OrderID']]
            if dropstat_type[row['OrderID']] == DROPSTAT_TYPE_COSTCUR:
                valid_cost_cur = min(row['CostCur'], order_chargeback[row['OrderID']], group_chargeback[group_id])
                if valid_cost_cur:
                    group_chargeback[group_id] -= valid_cost_cur
                    order_chargeback[row['OrderID']] -= valid_cost_cur
                    dropstat_costcur_requests_by_day[date][row['OrderID']] = valid_cost_cur
            else:
                valid_cost = min(row['Cost'], order_chargeback[row['OrderID']], group_chargeback[group_id])
                if valid_cost:
                    group_chargeback[group_id] -= valid_cost
                    order_chargeback[row['OrderID']] -= valid_cost
                    dropstat_eventcost_requests_by_day[date][row['OrderID']] = valid_cost

        charged_back_orders = self.safe_dropstat_request(dropstat_costcur_requests_by_day, DROPSTAT_TYPE_COSTCUR)
        charged_back_orders += self.safe_dropstat_request(dropstat_eventcost_requests_by_day, DROPSTAT_TYPE_EVENTCOST)
        charged_back_groups = {}

        for order_id, cost in charged_back_orders.items():
            charged_back_groups[order_id] = (cost, dropstat_type[order_id], group_id_by_order_id[order_id])

        return charged_back_groups

    def chargeback(self, yt, yql, yt_markov):
        request = yql.query('''
            PRAGMA yt.TmpFolder = "{tmp_path}";
            PRAGMA File('bigb.so', 'https://proxy.sandbox.yandex-team.ru/last/BIGB_UDF?attrs={{"released":"stable"}}');
            PRAGMA udf('bigb.so');

            $chargeBackBlackList = AsSet({chargeback_black_list});

            $chargeBackGroupOrders = SELECT
                *
            FROM (
                SELECT
                    Bigb::ParseOrderProfile(TableRow()) AS Profile
                FROM `{caesar_orders_table_path}`
            )
            WHERE
                (
                        Profile.Resources.DirectBannersLogFields.ContentType == "wallet"
                    OR
                        Profile.Resources.DirectBannersLogFields.GroupOrderID == 0
                )
                AND Profile.Resources.OrderMoney.GroupLastEventTime < {cur_time} - 60 * 60 * {last_event_time_delay}
                AND
                (
                    (
                            Profile.Resources.OrderMoney.GroupMoneyLeft < 0
                        AND
                            Profile.Resources.DirectBannersLogFields.CurrencyID == 0
                    )
                    OR
                    (
                            Profile.Resources.OrderMoney.GroupMoneyLeftCur < 0
                        AND
                            Profile.Resources.DirectBannersLogFields.CurrencyID != 0
                    )
                )
                AND NOT SetIncludes($chargeBackBlackList, AsSet(Profile.OrderID));

            $chargeBackOrders = SELECT
                *
            FROM (
                SELECT
                    Bigb::ParseOrderProfile(TableRow()) AS OrderProfile
                FROM `{caesar_orders_table_path}`
            ) AS CaesarOrders
            LEFT JOIN $chargeBackGroupOrders AS CaesarGroupOrders
            ON CaesarGroupOrders.Profile.OrderID = CaesarOrders.OrderProfile.Resources.DirectBannersLogFields.GroupOrderID
            WHERE
                OrderProfile.Resources.DirectBannersLogFields.ContentType != "wallet"
                AND OrderProfile.Resources.OrderMoney.OrderLastEventTime > {cur_time} - 60 * 60 * 24 * {chargeback_day_limit}
                AND NOT (OrderProfile.Resources.AutoBudget.AutoBudgetPaidActions ?? false)
                AND OrderProfile.Resources.DirectBannersLogFields.ContentType != "zen"
                AND NOT SetIncludes($chargeBackBlackList, AsSet(OrderProfile.OrderID))
                AND
                (
                    CaesarGroupOrders.Profile IS NOT NULL
                    OR
                    (
                        OrderProfile.Resources.DirectBannersLogFields.GroupOrderID == 0
                        AND
                        (
                            (
                                    OrderProfile.Resources.OrderMoney.OrderMoneyLeft < 0
                                AND
                                    OrderProfile.Resources.DirectBannersLogFields.CurrencyID == 0
                            )
                            OR
                            (
                                    OrderProfile.Resources.OrderMoney.OrderMoneyLeftCur < 0
                                AND
                                    OrderProfile.Resources.DirectBannersLogFields.CurrencyID != 0
                            )
                        )
                    )
                );

            $caesar_dump = SELECT
                OrderStat.OrderID AS OrderID,
                if(
                    OrderProfile.Resources.DirectBannersLogFields.GroupOrderID != 0,
                    OrderProfile.Resources.DirectBannersLogFields.GroupOrderID,
                    OrderProfile.OrderID
                ) as GroupOrderID,
                OrderStat.CostCur AS OrderCostCur,
                OrderStat.Cost AS OrderCost,
                OrderStat.LastEventTime AS LastEventTime,
                Profile.Resources.OrderMoney.GroupMoneyLeftCur ?? (OrderProfile.Resources.OrderMoney.OrderMoneyLeftCur ?? 0) AS GroupMoneyLeftCur,
                Profile.Resources.OrderMoney.GroupMoneyLeft ?? (OrderProfile.Resources.OrderMoney.OrderMoneyLeft ?? 0) As GroupMoneyLeft,
                OrderProfile.Resources.DirectBannersLogFields.ClientID as ClientID,
                OrderProfile.Resources.DirectBannersLogFields.CurrencyID != 0 as IsCurrencyOrder,
                OrderProfile as OrderProfile
            FROM $chargeBackOrders AS CaesarOrders
            LEFT JOIN `{order_stat_table_path}` AS OrderStat
            ON CaesarOrders.OrderProfile.OrderID = OrderStat.OrderID
            WHERE
                OrderStat.LastEventTime > {cur_time} - 60 * 60 * 24 * {chargeback_day_limit}
                AND OrderProfile.Resources.DirectBannersLogFields.ClientID ?? 0 not in ({chargeback_client_id_black_list})
                AND OrderStat.CostCur != 0
                AND OrderStat.Cost != 0;

            INSERT INTO `{caesar_dump_path}` WITH TRUNCATE
            SELECT
                *
            FROM $caesar_dump;

            SELECT
                *
            FROM $caesar_dump;

            SELECT
                sum(Profile.Resources.OrderMoney.GroupMoneyLeft ?? 0) as GroupMoneyLeft,
                sum(Profile.Resources.OrderMoney.GroupMoneyLeftCur ?? 0) as GroupMoneyLeftCur
            FROM $chargeBackGroupOrders
            WHERE Profile.Resources.OrderMoney.GroupLastEventTime > {cur_time} - 60 * 60 * 24 * {chargeback_day_limit};
        '''.format(
            tmp_path=self.Parameters.tmp_folder_path,
            order_stat_table_path=self.Parameters.order_stat_table_path,
            caesar_orders_table_path=self.Parameters.caesar_orders_prefix,
            cur_time=int(time.time()),
            last_event_time_delay=self.Parameters.last_event_time_delay,
            chargeback_black_list=', '.join(self.Parameters.chargeback_black_list),
            chargeback_client_id_black_list=', '.join(self.Parameters.chargeback_client_id_black_list),
            chargeback_day_limit=self.Parameters.chargeback_day_limit,
            caesar_dump_path='{}_{}'.format(self.Parameters.caesar_dump_path, datetime.now().strftime('%Y-%m-%d-%H:%M:%S')),
        ),
        syntax_version=1)

        request.run()
        logging.info('ChargeBack candidates {table_name} table: {yql_url}'.format(
            table_name=self.Parameters.history_table_path,
            yql_url=request.share_url,
        ))

        results = request.get_results()

        candidates = None
        negative_money_left_sum = None
        for i, res in enumerate(results):
            if i == 0:
                candidates = res
            elif i == 1:
                negative_money_left_sum = res
            else:
                break

        if negative_money_left_sum is not None:
            for row in negative_money_left_sum.get_iterator():
                parsed_row = dict(zip(negative_money_left_sum.column_names, row))

                self.solomon_client.set_value(
                    sensor='unfiltered_negative_money_left_cur_sum',
                    value=-parsed_row['GroupMoneyLeftCur'],
                )

                self.solomon_client.set_value(
                    sensor='unfiltered_negative_money_left_sum',
                    value=-parsed_row['GroupMoneyLeft'],
                )

                logging.info(
                    'Sent to Solomon:\n'
                    'Currency negative money left: {}\n'
                    'Chip negative money left: {}'
                    .format(parsed_row['GroupMoneyLeftCur'], parsed_row['GroupMoneyLeft'])
                )

        if not request.is_success:
            logging.info('\n'.join([str(err) for err in request.errors]))
            raise common.errors.TaskError("YQL Error. Check logs")

        if candidates is None:
            logging.info('No Orders to charge back')
            return

        start_time = int(time.time())

        old_chargeback_groups = yt_markov.select_rows('''
                GroupOrderID,
                max(StartTime) as StartTime,
                min(LastGroupMoneyLeftCur) as LastGroupMoneyLeftCur,
                min(LastGroupMoneyLeft) as LastGroupMoneyLeft
            from
                [{candidates_table_path}]
            where
                State = "{processing_state}"
            group by GroupOrderID
        '''.format(
            candidates_table_path=self.Parameters.history_table_path,
            processing_state=PROCESSING_STATE,
        ), input_row_limit=10**6)

        new_chargeback_groups = {}

        for row in candidates.get_iterator():
            candidate = dict(zip(candidates.column_names, row))
            new_chargeback_groups[candidate['GroupOrderID']] = candidate

        completed_chargeback_groups = set()
        chargeback_processing = set()
        longest_chargeback_execution = None

        for old_chargeback in old_chargeback_groups:
            if old_chargeback['GroupOrderID'] not in new_chargeback_groups or \
               -old_chargeback['LastGroupMoneyLeftCur'] < 1e6 or -old_chargeback['LastGroupMoneyLeft'] < 1e6:

                logging.info('GroupOrderID {} chargeback seems to be completed'.format(old_chargeback['GroupOrderID']))

                completed_chargeback_groups.add(old_chargeback['GroupOrderID'])
                continue
            else:
                new_group_to_be_charged_back = new_chargeback_groups[old_chargeback['GroupOrderID']]

                is_enough_time_since_dropstat_request = old_chargeback['StartTime'] < start_time - 60 * 60 * self.Parameters.minimum_chargeback_waiting_time
                is_any_charge_back = (new_group_to_be_charged_back['IsCurrencyOrder'] and old_chargeback['LastGroupMoneyLeftCur'] != new_group_to_be_charged_back['GroupMoneyLeftCur']) or \
                    (not new_group_to_be_charged_back['IsCurrencyOrder'] and old_chargeback['LastGroupMoneyLeft'] != new_group_to_be_charged_back['GroupMoneyLeft'])

                if is_enough_time_since_dropstat_request and is_any_charge_back:
                    logging.info('GroupOrderID {} chargeback seems to be completed'.format(old_chargeback['GroupOrderID']))

                    completed_chargeback_groups.add(old_chargeback['GroupOrderID'])
                    continue

            logging.info('GroupOrderID {} chargeback is still processing'.format(old_chargeback['GroupOrderID']))

            if longest_chargeback_execution is None or longest_chargeback_execution > old_chargeback['StartTime']:
                longest_chargeback_execution = old_chargeback['StartTime']
            chargeback_processing.add(old_chargeback['GroupOrderID'])

        if longest_chargeback_execution is None:
            longest_chargeback_execution = 0
        else:
            longest_chargeback_execution = start_time - longest_chargeback_execution

        self.solomon_client.set_value(
            sensor='dropstat_delay',
            value=longest_chargeback_execution
        )

        logging.info('Sent to Solomon: Longest chargeback execution is {}s'.format(longest_chargeback_execution))

        update_old_chargebacks = []

        old_chargebacks = yt_markov.select_rows('''
                *
            from
                [{candidates_table_path}]
            where
                State = "{processing_state}"
        '''.format(
            candidates_table_path=self.Parameters.history_table_path,
            processing_state=PROCESSING_STATE,
        ), input_row_limit=10**6)

        for old_chargeback in old_chargebacks:
            if old_chargeback['GroupOrderID'] in completed_chargeback_groups:
                old_chargeback['State'] = COMPLETE_STATE
                update_old_chargebacks.append(old_chargeback)

        yt_markov.insert_rows(self.Parameters.history_table_path, update_old_chargebacks, update=True)

        approved_candidates = {}
        for row in candidates.get_iterator():
            candidate = dict(zip(candidates.column_names, row))
            if candidate['GroupOrderID'] not in chargeback_processing:
                candidate['StartTime'] = start_time
                approved_candidates[candidate['OrderID']] = candidate

        self.check_chargeback_limit(approved_candidates.values())

        charged_back = self.prepare_dropstat_request(yt, list(approved_candidates.values()))

        start_time = int(time.time())
        fresh_charged_back_orders = []
        for order_id, (cost, dropstat_type, group_order_id) in charged_back.items():
            fresh_charged_back_orders.append({
                'GroupOrderID': group_order_id,
                'OrderID': order_id,
                'StartTime': start_time,
                'CostCur': cost if dropstat_type == DROPSTAT_TYPE_COSTCUR else 0,
                'Cost': cost if dropstat_type == DROPSTAT_TYPE_EVENTCOST else 0,
                'State': PROCESSING_STATE,
                'LastGroupMoneyLeftCur': approved_candidates[order_id]['GroupMoneyLeftCur'],
                'LastGroupMoneyLeft': approved_candidates[order_id]['GroupMoneyLeft'],
            })

        yt_markov.insert_rows(self.Parameters.history_table_path, fresh_charged_back_orders)

        processing_count = next(yt_markov.select_rows('''
                sum(if(CostCur != 0, 1, 0)) as CostCurCount,
                sum(if(Cost != 0, 1, 0)) as EventCostCount
            from
                [{candidates_table_path}]
            where
                State = "{processing_state}"
            group by 1
        '''.format(
            candidates_table_path=self.Parameters.history_table_path,
            processing_state=PROCESSING_STATE,
        ), input_row_limit=10**6))

        self.solomon_client.set_value(
            sensor='orders_count',
            value=processing_count['CostCurCount'],
            labels={
                'dropstat_type': DROPSTAT_TYPE_COSTCUR,
                'state': 'processing',
            },
        )

        self.solomon_client.set_value(
            sensor='orders_count',
            value=processing_count['EventCostCount'],
            labels={
                'dropstat_type': DROPSTAT_TYPE_EVENTCOST,
                'state': 'processing',
            },
        )

        logging.info(
            'Sent to Solomon:\n'
            'CostCur dropstat requests count: {}\n'
            'EventCost dropstat requests count: {}'
            .format(processing_count['CostCurCount'], processing_count['EventCostCount'])
        )

        return request.is_success

    def on_execute(self):
        from solomon import PushApiReporter, OAuthProvider

        from yt.wrapper import YtClient
        from yql.api.v1.client import YqlClient

        self.yt_token = self.Parameters.yabs_cs_chback_secret_id.data()['yt-token']
        self.yql_token = self.Parameters.yabs_cs_chback_secret_id.data()['yql-token']
        self.solomon_token = self.Parameters.yabs_cs_chback_secret_id.data()['solomon-token']
        self.dropstat_token = self.Parameters.dropstat_secret_id.data()['dropstat_token']

        yt = YtClient(proxy=self.Parameters.cluster, token=self.yt_token)
        yt_markov = YtClient(proxy='markov', token=self.yt_token)
        yql = YqlClient(db=self.Parameters.cluster, token=self.yql_token)

        self.solomon_client = PushApiReporter(
            project='yabscs_chargeback' if self.Parameters.testing_mode else 'yabscs',
            cluster='chargeback',
            service='chargeback',
            url=SOLOMON_API_URL,
            auth_provider=OAuthProvider(self.solomon_token),
        )

        self.chargeback(yt, yql, yt_markov)
