# -*- coding: utf-8 -*-

import copy
import datetime
import requests
import logging
import time
from collections import defaultdict

from dateutil.tz import gettz as get_timezone
from sandbox.projects.yabs.base_bin_task import BaseBinTask

from sandbox import common, sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2 import parameters
from .config import CONFIG, YT_PREFIX
from .query import YQL_QUERY


class YabsDoubleClick(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        kill_timeout = 3600 * 12
        description = "Double click data maker task"

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = sdk2.parameters.Dict("Filter resource by", default={"name": "YabsDoubleClick"})

        with parameters.Group("Manual run parameters") as manual_params:
            manual_date_enabled = sdk2.parameters.Bool("Calculate on specific date", default=False)
            manual_date = sdk2.parameters.String("Specific date, Format YYYY-MM-DD", default="")

        with parameters.RadioGroup("Profile") as profile:
            profile.values["testing"] = profile.Value(value="testing", default=True)
            profile.values["production"] = profile.Value(value="production")

        with parameters.Group("YQL parameters") as yql_params:
            retry_period = sdk2.parameters.Integer("Time period to recheck yql completeness, seconds", default=60 * 15)
            wait_data_period = sdk2.parameters.Integer("Time period to recheck JoinedEFH data availability, seconds", default=60 * 60)

    def email_notification(self):
        logging.info("Sending email")
        hold_dict = defaultdict(lambda: defaultdict(list))
        sorted_notification_data = sorted(self._email_notification_data, key=lambda x: x[0]['clicks'], reverse=True)
        for row, remaining in sorted_notification_data:
            login = row["login"]
            tier_type = self.gray_list[login]["tier_type"]
            self.hold_stat[tier_type] += 1
            hold_dict[tier_type][login].append('<li>На {pageid}-{impid} ({url}, "{block_caption}", quality {coef:5.2f} '
                                               'last day clicks: {clicks} lclicks: {lclicks}) '
                                               'включение ожидается {estimated_date}</li>'.format(
                pageid=row["pageid"],
                impid=row["impid"],
                block_caption=row["block_caption"],
                url=row["name"],
                estimated_date=self.offset_date(self.current_day, remaining),
                clicks=row["clicks"] * 10,
                lclicks=row["lclicks"] * 10,
                coef=row["quality_coef"]
            ))

        for tier_type in hold_dict:
            body = "<p>Блоки, у которых скоро включится подтверждение перехода:</p>"
            for login in hold_dict[tier_type]:
                body += "<p>Логин {}</p>".format(login)
                body += "<ul>{blocks}</ul>".format(blocks='\n'.join(hold_dict[tier_type][login]))
            body += """<p>Подробные данные можно увидеть в
<a href='https://yt.yandex-team.ru/hahn/navigation?path=//home/yabs/double-click/production/data&'>таблице</a></p>
"""
            logging.info("Email body: %s", body)
            if self.profile == 'testing':
                address = self.config.get("EMAIL_RECIPIENTS_TESTING")
            else:
                address = ["double-click-{}@yandex-team.ru".format(tier_type), ]
            logging.info("Email address: %s", address)
            self.server.notification(
                subject="[DoubleClick] {day} blocks list".format(day=self.current_day),
                body=body,
                recipients=address,
                transport=common.types.notification.Transport.EMAIL,
                urgent=False,
                type='html'
            )
            logging.info("Email to {} sent".format(tier_type))

    def pi_request(self, notify_type, login, pageid, block_ids):
        json_data = {
            'login': login,
            'notification': notify_type,
            'custom_data': {
                'page_id': pageid,
                'block_ids': list(block_ids)
            }
        }
        if self.profile == 'testing':
            logging.info('Try to request')
            logging.info(json_data)
        else:
            resp = requests.post(self.config["PI_URL"] + self.config["PI_HANDLER"], json=json_data)
            resp.raise_for_status()
            logging.info('Request to {} about {}({}) sent'.format(login, pageid, ','.join(block_ids)))

    def pi_notification(self):
        for (pageid, notify_type), data_list in self._pi_notification_data.items():
            if data_list:
                login = data_list[0]["login"]
                blocks = []
                for row in data_list:
                    blocks.append('{}-{}'.format(pageid, row['impid']))
                self.pi_request(notify_type, login, int(pageid), set(blocks))

    def get_tokens(self):
        logging.info("Getting tokens")
        # https://yav.yandex-team.ru/secret/sec-01ebpxe46rv3wh8rkw90t2f58h/explore/versions
        secret = sdk2.yav.Secret(
            "sec-01ebpxe46rv3wh8rkw90t2f58h"
        ).data()
        self.yt_token = secret["yt_token"]
        self.yql_token = secret["yql_token"]
        self.solomon_token = secret["solomon_token"]

    def run_yql(self):
        yql_query = YQL_QUERY.format(
            current_day=self.current_day,
            experiment=self.config["EXPERIMENT"],
            profile=self.profile,
            prefix=YT_PREFIX,
            expiration_interval=self.config["EXPIRATION_INTERVAL"],
        )
        logging.info("Main YQL:\n%s", yql_query)
        request = self.yql_client.query(yql_query, syntax_version=1)
        request.run()
        logging.info("Operation id %s", str(request.operation_id))
        self.Context.yql_operation_id = request.operation_id

    def wait_yql(self):
        from yql.client.operation import YqlOperationStatusRequest

        status = YqlOperationStatusRequest(self.Context.yql_operation_id)
        status.run()
        if status.status in status.IN_PROGRESS_STATUSES:
            logging.info("yql query still running")
            raise sdk2.WaitTime(self.Parameters.retry_period)
        if not status.is_success:
            if status.errors:
                logging.error("YQL query returned errors:")
                for error in status.errors:
                    logging.error("- %s", str(error))
            raise TaskFailure("YQL query failed")

    def _get_daily_table_path(self, day):
        return "{prefix}/{profile}/raw-data/{current_day}".format(
            prefix=YT_PREFIX, profile=self.profile, current_day=day,
        )

    def _get_main_table_path(self):
        return "{prefix}/{profile}/data".format(prefix=YT_PREFIX, profile=self.profile)

    def wait_data(self):
        logging.info("Check for logs available (JoinedEFH)")
        if not self.yt_hahn.exists(
            "//home/bs/logs/JoinedEFH/1h/{current_day}T23:00:00".format(current_day=self.current_day)
        ):
            logging.info("Not all JoinedEFH data are available.")
            raise sdk2.WaitTime(self.Parameters.wait_data_period)
        logging.info("All data available")

    @staticmethod
    def get_row_key(row):
        return row["pageid"], row["impid"], row["pagedomainmd5"]

    def get_data(self):
        logging.info("Get data from YQL date=%s", self.current_day)
        self.rows = self.yt_hahn.read_table(self._get_daily_table_path(self.current_day))

        self.prev_rows = []
        for index in range(1, self.config["CALCULATE_DAYS"]):
            day = self.offset_date(self.current_day, -1 * index)
            logging.info("Get data from YQL date=%s", day)
            rows = self.yt_hahn.read_table(self._get_daily_table_path(day))
            self.prev_rows.append({
                self.get_row_key(row): {"lclicks": row["lclicks"], "clicks": row["clicks"]}
                for row in rows
            })
        logging.info("Get previous day info")
        result = self.yt_hahn.select_rows(
            "* FROM [{table}] WHERE date='{day}'".format(table=self._get_main_table_path(), day=self.last_day)
        )
        self.prev_data = {
            self.get_row_key(row): {
                key: row[key]
                for key in ["on_hold_days", "status", "enabled", "last_date_status_change"]
            }
            for row in result
        }
        logging.info("Got %d rows from %s", len(self.prev_data), self._get_main_table_path())

        result = self.yt_hahn.read_table(self.config["YT_HOLD_TABLE_PATH"])
        self.gray_list = {
            row["partner_login"]: {"tier_type": row["tier_type"]}
            for row in result
        }
        logging.info("Got %d rows from %s", len(self.gray_list), self.config["YT_HOLD_TABLE_PATH"])

    @staticmethod
    def cost_none_check(cost):
        if cost is None:
            cost = 0
        return cost

    def _add_data_row(self, data, row, status, enabled=False, **kwargs):
        self.total_yql_rows += 1
        self.status_stat[status] += 1
        prev_data = self.prev_data.get(self.get_row_key(row), {})
        result = copy.deepcopy(row)
        last_date_status_change = prev_data.get("last_date_status_change", self.current_day)
        if prev_data and enabled != prev_data["enabled"]:
            last_date_status_change = self.current_day
        if enabled:
            self.total_enabled['all'] += 1
            self.total_cost['all'] += self.cost_none_check(row["cost_rub"])
            if row["is_ssp"]:
                self.total_enabled['ssp'] += 1
                self.total_cost['ssp'] += self.cost_none_check(row["cost_rub"])
            elif row["block_model"] in ('mobile_app_rtb', 'internal_mobile_app_rtb'):
                self.total_enabled['sdk'] += 1
                self.total_cost['sdk'] += self.cost_none_check(row["cost_rub"])
            else:
                self.total_enabled['rsya'] += 1
                self.total_cost['rsya'] += self.cost_none_check(row["cost_rub"])
        if prev_data.get("enabled") and not enabled:
            self.today_disabled += 1
            self._add_to_pi_notification_data("PI_DISABLE_TYPE", row)
        if not prev_data.get("enabled") and enabled:
            self.today_enabled += 1
            self._add_to_pi_notification_data("PI_ENABLE_TYPE", row)
        result.update(
            {
                "date": self.current_day,
                "status": status,
                "enabled": enabled,
                "on_hold_days": kwargs.get("on_hold_days", 0),
                "last_date_status_change": last_date_status_change,
                "params": self.config["ADDITIONAL_JSON"],
            }
        )
        if (row["pageid"], row["impid"]) in self.config["ADDITIONAL_JSON_CUSTOM"].keys():
            result.update({"params": self.config["ADDITIONAL_JSON_CUSTOM"].get((row["pageid"], row["impid"]))})
        result.pop("is_turbo")
        result.pop("is_native_widget")
        data.append(result)

    def _add_to_pi_notification_data(self, notify_type, row):
        key = (row["pageid"], self.config[notify_type])
        self._pi_notification_data[key].append(row)

    def main_logic(self):
        logging.info("Main logic start")
        table_path = self._get_main_table_path()
        data = []
        for row in self.rows:
            if len(data) > 99999:
                logging.info("Save %d rows to %s", len(data), table_path)
                self.yt_markov.insert_rows(table_path, data)
                data = []

            pageid = row["pageid"]
            previous_day = self.prev_data.get(self.get_row_key(row), {})

            if pageid in self.config["BLACK_LIST"]:
                self._add_data_row(data, row, status="pageid in blacklist", enabled=True)
                continue

            if pageid in self.config["WHITE_LIST"]:
                self._add_data_row(data, row, status="pageid in whitelist")
                continue

            if (pageid, row["impid"]) in self.config["BLACK_LIST_IMP"]:
                self._add_data_row(data, row, status="page-imp in blacklist", enabled=True)
                continue

            if (pageid, row["impid"]) in self.config["WHITE_LIST_IMP"]:
                self._add_data_row(data, row, status="page-imp in whitelist")
                continue

            if row["clicks"] < self.config["CLICKS_THRESHOLD"]:
                key = self.get_row_key(row)
                lclicks, clicks = row["lclicks"], row["clicks"]
                for prev_row in self.prev_rows:
                    lclicks += prev_row.get(key, {"lclicks": 0})["lclicks"]
                    clicks += prev_row.get(key, {"clicks": 0})["clicks"]
                    if clicks >= self.config["CLICKS_THRESHOLD"]:
                        break

                row["quality_coef"] = 0.0 if clicks == 0 else 100.0 * lclicks / float(clicks)

                if clicks < self.config["CLICKS_THRESHOLD"]:
                    self._add_data_row(data, row, status="not enough clicks", enabled=previous_day.get("enabled", False))
                    continue

            if row["block_model"] in ('mobile_app_rtb', 'internal_mobile_app_rtb') and row['is_native_widget'] is True:
                self._add_data_row(data, row, status="block is native widget mobile_app_rtb")
                continue

            if row['is_turbo'] is True:
                self._add_data_row(data, row, status="block is turbo")
                continue

            if previous_day.get("enabled") and row["quality_coef"] >= self.config["OLD_PAGEID_QUALITY_THRESHOLD"]:
                self._add_data_row(data, row, status="good quality first day")
                continue

            if not previous_day.get("enabled") and row["quality_coef"] >= self.config["NEW_PAGEID_QUALITY_THRESHOLD"]:
                self._add_data_row(data, row, status="good quality")
                continue

            if row["login"] in self.gray_list:
                if not previous_day:
                    self._add_data_row(
                        data,
                        row,
                        status="hold, {limit} days left".format(limit=self.config["MAX_HOLD_DAYS"]),
                        on_hold_days=1,
                    )
                    self._email_notification_data.append((row, self.config["MAX_HOLD_DAYS"]))
                    continue

                if previous_day["enabled"] is False and previous_day["on_hold_days"] < self.config["MAX_HOLD_DAYS"]:
                    self._add_data_row(
                        data,
                        row,
                        status="hold, {limit} days left".format(
                            limit=self.config["MAX_HOLD_DAYS"] - previous_day["on_hold_days"]
                        ),
                        on_hold_days=previous_day["on_hold_days"] + 1,
                    )
                    self._email_notification_data.append((
                        row,
                        self.config["MAX_HOLD_DAYS"] - previous_day["on_hold_days"]
                    ))
                    continue

                if previous_day["enabled"] is False:
                    self._add_data_row(data, row, status="from hold", enabled=True)
                    continue

            threshold = self.config["NEW_PAGEID_QUALITY_THRESHOLD"]
            if previous_day.get("enabled"):
                threshold = self.config["OLD_PAGEID_QUALITY_THRESHOLD"]
            self._add_data_row(
                data,
                row,
                status="quality_coef < {threshold}".format(threshold=threshold),
                enabled=True,
            )

        logging.info("Save %d rows to %s", len(data), table_path)
        self.yt_markov.insert_rows(table_path, data)
        self.yt_markov.set("{table_path}/@max_unix_timestamp".format(table_path=table_path), int(time.time()))

    def create_clients(self):
        from yql.api.v1.client import YqlClient
        import yt.wrapper as yt

        self.yql_client = YqlClient(db="hahn", token=self.yql_token)
        self.yt_markov = yt.YtClient(proxy="markov", token=self.yt_token)
        self.yt_hahn = yt.YtClient(proxy="hahn", token=self.yt_token)

    @staticmethod
    def offset_date(date_str, offset_days):
        date = datetime.datetime.strptime(date_str, '%Y-%m-%d')
        date += datetime.timedelta(days=offset_days)
        return date.strftime("%Y-%m-%d")

    def set_dates(self):
        if self.Parameters.manual_date_enabled:
            logging.info("Manual run")
            self.current_day = self.Parameters.manual_date
            self.last_day = self.offset_date(self.current_day, -1)
        else:
            now = datetime.datetime.now(tz=get_timezone("Europe/Moscow"))
            last_day = (now - datetime.timedelta(days=1)).strftime("%Y-%m-%d")

            row = next(self.yt_hahn.select_rows("""
                date
            from
                [{table}]
            order by
                date desc
            limit 1
        """.format(table=self._get_main_table_path()), input_row_limit=100000000), None)

            if row is None:
                self.current_day = last_day
                self.last_day = self.offset_date(self.current_day, -1)
            else:
                self.last_day = row["date"]
                self.current_day = self.offset_date(row["date"], 1)

        logging.info("Current day: %s", self.current_day)
        logging.info("Last day: %s", self.last_day)

    def init(self):
        logging.info("Init")
        self.profile = self.Parameters.profile
        logging.info("Profile: %s", self.profile)
        self.config = CONFIG
        self._email_notification_data = []
        self._pi_notification_data = defaultdict(list)
        self.total_yql_rows = 0
        self.total_cost = defaultdict(int)
        self.total_enabled = defaultdict(int)
        self.status_stat = defaultdict(int)
        self.hold_stat = defaultdict(int)
        self.today_enabled = 0
        self.today_disabled = 0

        logging.info("---- Config ----")
        logging.info("YT_PREFIX: %s", YT_PREFIX)
        for key, value in self.config.items():
            logging.info("%s = %s", key, value)
        logging.info("---- End of config ----")

    def push_to_solomon(self):
        import requests
        import json

        logging.info("Sending sensors to solomon")
        headers = {
            "Content-Type": "application/json",
            "Authorization": "OAuth {solomon_token}".format(solomon_token=self.solomon_token),
        }
        data = {
            "sensors": [
                {
                    "labels": {
                        "profile": self.profile,
                        "sensor": sensor,
                    },
                    "value": value,
                }
                for sensor, value in [
                    ("total_yql_rows", self.total_yql_rows),
                    ("total_cost", self.total_cost['all']),
                    ("total_enabled", self.total_enabled['all']),
                    ("today_enabled", self.today_enabled),
                    ("today_disabled", self.today_disabled),
                ] + [
                    ("status_" + key.replace(" ", "_"), value) for key, value in self.status_stat.items()
                ] + [
                    ("hold_" + key.replace(" ", "_"), value) for key, value in self.hold_stat.items()
                ] + [
                    ("total_cost_" + key, value) for key, value in self.total_cost.items()
                ] + [
                    ("total_enabled_" + key, value) for key, value in self.total_enabled.items()
                ]
            ]
        }
        answer = requests.post(
            "https://solomon.yandex.net/api/v2/push?project=yabs&service=double_click&cluster=yabs",
            data=json.dumps(data),
            headers=headers,
        ).json()
        logging.info("Solomon response: %s", answer)
        if answer.get("status", None) != "OK":
            logging.error("push_to_solomon() failed")

    def on_execute(self):
        self.init()
        self.get_tokens()
        self.create_clients()
        self.set_dates()
        self.wait_data()
        if not self.yt_hahn.exists(self._get_daily_table_path(self.current_day)):
            with self.memoize_stage.run_yql():
                self.run_yql()
            self.wait_yql()
        self.get_data()
        self.main_logic()
        if self._email_notification_data:
            self.email_notification()
        if self._pi_notification_data:
            self.pi_notification()
        self.push_to_solomon()
