import logging

from collections import defaultdict

from sandbox import common
from sandbox.sdk2 import parameters
from sandbox.projects.yabs.dropstat.base import BaseDropStatTask
from sandbox.projects.yabs.dropstat.base.config import EMAIL_RECIPIENTS, EMAIL_RECIPIENTS_PRE

CHECK_QUERY = '''
pragma yt.ForceInferSchema;

select count(*) from
{request_path} as req
left join `{archive_path}` as arc
on req.{id_field} = arc.{id_field}
where req.dropstattimestamp > coalesce(arc.dropstattimestamp, 0)
'''

SUM_QUERY = '''
pragma yt.ForceInferSchema;

select {sum_cols}
from {path}
'''

ST_COMMENT_TEXT = '''
Dropstat request for log {log_name} complete: group {group_id}. Totals:
{totals}
'''

ST_EMPTY_COMMENT_TEXT = '''
Dropstat request for log {log_name} complete: group {group_id}.
No undo logs created.
'''

EMAIL_BODY = '''
Dropstat request for ticket {ticket} exceeds limits: {col} = {value} > {min_value}.

Ticket link: https://st.yandex-team.ru/{ticket}
'''


class YabsDropStatArchive(BaseDropStatTask):
    '''Archive dropstat requests and send stats to tickets
    '''
    class Parameters(BaseDropStatTask.Parameters):
        description = 'Archive dropstat requests and send stats to tickets'

        with BaseDropStatTask.Parameters.exec_params() as exec_params:
            st_token = parameters.YavSecret(
                'Startrek token secret',
                required=True,
            )

            wait_archive = parameters.Bool("Wait archive delivery", default=True)

    def get_done_group_requests(self):
        from yabs.stat.dropstat2.pylibs.common.request import DropStatRequest

        rows = self.yt_meta_client.select_rows("* from [{}]".format(self.meta_path))
        requests = []
        for row in rows:
            try:
                request = DropStatRequest.from_yt_row(row, log_type=self.Parameters.log_type)
            except Exception:
                logging.warning('Failed to create request object from row %s', row)
            else:
                requests.append(request)

        group_requests = defaultdict(list)
        for req in requests:
            group_requests[req.group_id].append(req)

        result = {}
        for group_id, requests in group_requests.items():
            if all([req.state in ('done', 'skipped') for req in requests]):
                result[group_id] = requests

        logging.info('Got requests: %s', result)
        return result

    def run_yql_query(self, query):
        from yql.api.v1.client import YqlClient

        logging.info('Executing YQl query: %s', query)

        yql_client = YqlClient(db=self.Parameters.work_yt_proxy, token=self.yt_token)
        request = yql_client.query(query, syntax_version=1)
        request.run()

        request_url = request.share_url
        if request_url is not None:
            logging.info('Operation request url: %s', request_url)
        else:
            logging.warning('No operation url available')

        results = request.get_results()

        if not request.is_success:
            raise RuntimeError('YQL query failed')

        for table in results:
            for row in table.get_iterator():
                return row

    def check_empty(self, requests):
        tables = ['{}/merged/{}'.format(self.work_dir, req.request_id) for req in requests]
        for table in tables:
            if self.yt_client.get(table + '/@row_count', 0) > 0:
                return False
        return True

    def concat_tables(self, requests):
        tables = ['`{}/merged/{}`'.format(self.work_dir, req.request_id) for req in requests]
        tables = filter(lambda table: self.yt_client.get(table[1:-1] + '/@row_count', 0) > 0, tables)
        return 'CONCAT({})'.format(','.join(tables))

    def check_done_requests(self, requests):
        path = self.concat_tables(requests)
        query = CHECK_QUERY.format(
            request_path=path,
            archive_path=self.log_description.archive_path + self.archive_suffix,
            id_field=self.log_description.id_column,
        )

        return self.run_yql_query(query)[0] == 0

    def calculate_totals(self, requests):
        sum_cols = ','.join('{} as {}'.format(expr, col) for col, expr in self.log_description.totals_columns)
        path = self.concat_tables(requests)

        query = SUM_QUERY.format(
            sum_cols=sum_cols,
            path=path,
        )

        col_names = (c[0] for c in self.log_description.totals_columns)
        return dict(zip(col_names, self.run_yql_query(query)))

    def create_st_comment(self, ticket, group_id, totals, add_access):
        from startrek_client import Startrek

        if not ticket:
            return

        if totals:
            totals_text = ', '.join('{} {}'.format(col, value) for col, value in totals.items())
            text = ST_COMMENT_TEXT.format(
                log_name=self.Parameters.log_type,
                group_id=group_id,
                totals=totals_text,
            )
        else:
            text = ST_EMPTY_COMMENT_TEXT.format(
                log_name=self.Parameters.log_type,
                group_id=group_id,
            )

        if not self.Parameters.production:
            text = '[TESTING]\n' + text

        st_token = self.Parameters.st_token.data()['startrek_token']
        st_client = Startrek(useragent='python', token=st_token)
        issue = st_client.issues[ticket]

        if len(list(issue.comments.get_all())) < 1000:
            issue.comments.create(text=text)

        if add_access:
            # Reload to fix "Conflict: Issue has already been modified."
            issue = st_client.issues[ticket]
            access = set([user.login for user in issue.access] + self.recipients)
            issue.update(access=list(access))

    def send_email_if_needed(self, ticket, totals):
        if not self.recipients:
            return False

        for col, min_value in self.log_description.min_notification_values.items():
            if totals[col] > min_value:
                logging.info('Sending email notification')
                body = EMAIL_BODY.format(ticket=ticket, col=col, value=totals[col], min_value=min_value)
                self.server.notification(
                    subject="Dropstat: large request detected",
                    body=body,
                    recipients=self.recipients,
                    transport=common.types.notification.Transport.EMAIL,
                    urgent=True,
                )
                return True

        return False

    def move_to_archive(self, requests):
        with self.yt_meta_client.Transaction(type='tablet'):
            keys = [{'RequestID': req.request_id} for req in requests]
            logging.info('Deleting keys: %s', keys)
            self.yt_meta_client.delete_rows(self.meta_path, keys)
            rows = [req.to_yt_row() for req in requests]
            logging.info('Insert rows: %s', rows)
            self.yt_meta_client.insert_rows(self.work_dir + '/Archive', rows)

    def on_execute(self):
        self.recipients = EMAIL_RECIPIENTS if self.Parameters.production else EMAIL_RECIPIENTS_PRE

        for group_id, requests in self.get_done_group_requests().items():
            done_requests = filter(lambda req: req.state == 'done', requests)

            if len(done_requests) == 0 or self.check_empty(done_requests):
                logging.info('Group %s has 0 total rows', group_id)
                self.create_st_comment(requests[0].st_task, group_id, None, False)
                self.move_to_archive(requests)
                continue

            if self.Parameters.production or self.Parameters.wait_archive:
                if not self.check_done_requests(done_requests):
                    logging.info('Group %s is not finished yet', group_id)
                    continue
            else:
                logging.info('Archive delivery check skipped for group %s', group_id)

            logging.info('Archiving group %s', group_id)
            ticket = done_requests[0].st_task

            if ticket:
                totals = self.calculate_totals(done_requests)
                sent_email = self.send_email_if_needed(ticket, totals)
                self.create_st_comment(ticket, group_id, totals, sent_email)
            self.move_to_archive(requests)
