import logging

from sandbox.common.types.resource import State
from sandbox.common.errors import TaskFailure
from sandbox.sdk2 import (
    Resource,
    ResourceData,
    Task,
    parameters,
    service_resources
)

from sandbox.projects.yabs.dropstat.resources import YabsDropStatBinaryPackage

from .config import BASE_DIRECTORY


class BaseDropStatTask(Task):
    _yt_client = None
    _yt_meta_client = None

    class Requirements(Task.Requirements):
        cores = 1
        ram = 2048
        disk_space = 2048

        class Caches(Task.Requirements.Caches):
            pass

    class Parameters(Task.Parameters):

        with parameters.Group('Execution') as exec_params:
            production = parameters.Bool('Production')
            log_type = parameters.String('Log type', required=True)

        with parameters.Group('YT') as yt_params:
            meta_yt_proxy = parameters.String('Meta YT cluster')
            work_yt_proxy = parameters.String('Working YT cluster', required=True)
            yt_token_secret = parameters.YavSecret(
                'YT token secret',
                description='Required key: yt_token',
                required=True
            )

    def on_save(self):
        res = self.get_last_resource(service_resources.SandboxTasksBinary, {'name': 'YabsDropStat'})

        logging.info('Using task resource #%s', res.id)
        self.Requirements.tasks_resource = res.id

    def on_prepare(self):
        from yabs.stat.dropstat2.pylibs.common.log_description import LOGS_MAP

        self.log_description = LOGS_MAP[self.Parameters.log_type]
        self.yt_token = self.Parameters.yt_token_secret.data()['yt_token']

        release_dir = '/production/' if self.Parameters.production else '/prestable/'
        self.work_dir = BASE_DIRECTORY + release_dir + self.Parameters.log_type
        self.meta_path = self.work_dir + '/Queue'
        self.meta_path_archive = self.work_dir + '/Archive'

        self.archive_suffix = '' if self.Parameters.production else 'Prestable'

    def get_last_resource(self, resource_type, attrs_upd=None):
        if attrs_upd is None:
            attrs_upd = {}
        attrs = {'released': 'stable'}
        attrs.update(attrs_upd)
        stable_res = Resource.find(resource_type=resource_type,
                                   state=State.READY,
                                   owner='YABS_DROP_STAT',
                                   attrs=attrs).order(-Resource.id).first()
        attrs['released'] = 'prestable'
        pre_res = Resource.find(resource_type=resource_type,
                                state=State.READY,
                                owner='YABS_DROP_STAT',
                                attrs=attrs).order(-Resource.id).first()

        if self.Parameters.production or getattr(stable_res, 'id', 0) > getattr(pre_res, 'id', 0):
            res = stable_res
        else:
            res = pre_res

        return res

    @property
    def yt_client(self):
        if self._yt_client is None:
            import yt.wrapper as yt

            self._yt_client = yt.YtClient(proxy=self.Parameters.work_yt_proxy, token=self.yt_token)

        return self._yt_client

    @property
    def yt_meta_client(self):
        if self._yt_meta_client is None:
            import yt.wrapper as yt

            self._yt_meta_client = yt.YtClient(proxy=self.Parameters.meta_yt_proxy, token=self.yt_token, config={'backend': 'rpc'})

        return self._yt_meta_client

    def find_drop_requests(self, request_ids):
        from yabs.stat.dropstat2.pylibs.common.request import DropStatRequest
        selected = list(self.yt_meta_client.lookup_rows(self.meta_path, [{'RequestID': int(id)} for id in request_ids]))
        for r in selected:
            logging.info('Request selected: %s', r)
        if len(selected) != len(request_ids):
            raise TaskFailure('Not all requests was found.')

        return [DropStatRequest.from_yt_row(row, log_type=self.Parameters.log_type) for row in selected]

    def get_binary_path(self):
        bin_res = self.get_last_resource(YabsDropStatBinaryPackage)

        logging.info('Using binary resource #%s', bin_res.id)
        return str(ResourceData(bin_res).path)
