import logging
import os

from sandbox.sdk2 import parameters

from sandbox.projects.yabs.dropstat.base import BaseDropStatTask
from sandbox.projects.yabs.dropstat.base.config import YTPUMP_JOBS


class YabsDropStatMerge(BaseDropStatTask):
    '''Selects data for dropstat request
    '''
    class Parameters(BaseDropStatTask.Parameters):
        description = 'Prepare dropstat request'

        with BaseDropStatTask.Parameters.exec_params() as exec_params:
            test_run = parameters.Bool('Testing mode - merge tables to tmp dir')

    def on_execute(self):
        from yabs.stat.dropstat2.pylibs.merge.run import run_merge
        from yabs.stat.dropstat2.pylibs.common.request import DropStatRequest

        work_dir = self.work_dir + '/merge_tmp'
        merged_dir = self.work_dir + '/merged'
        selected_dir = self.work_dir + '/prepare'

        # cleanup tables in case something went wrong
        with self.yt_client.Transaction():
            # last iteration failed
            tmp_tables = self.yt_client.list(work_dir, absolute=True)

            # "prepare" stage created duplicate tables
            merged_tables = set(self.yt_client.list(merged_dir))
            selected_tables = set(self.yt_client.list(selected_dir))
            extra_tables = merged_tables.intersection(selected_tables)
            extra_tables = ['{}/{}'.format(selected_dir, table) for table in extra_tables]

            if tmp_tables:
                logging.warning('Lost tmp tables from last iteration: %s', tmp_tables)
            if extra_tables:
                logging.warning('Extra selected tables: %s', extra_tables)
            for table in tmp_tables + extra_tables:
                self.yt_client.remove(table)

            # Delete leftovers from the previously finished operations
            all_merged_tables = self.yt_client.list(merged_dir)
            requests_rows = self.yt_meta_client.lookup_rows(self.meta_path, [{'RequestID': int(t)} for t in all_merged_tables])
            requests = [r['RequestID'] for r in requests_rows]

            archive_requests_rows = self.yt_meta_client.lookup_rows(self.meta_path_archive, [{'RequestID': int(t)} for t in all_merged_tables])
            links_dict = {}
            for row in archive_requests_rows:
                request = DropStatRequest.from_yt_row(row, self.Parameters.log_type)
                if request.params.link_node:
                    links_dict[str(request.request_id)] = '{}/{}'.format(request.params.link_node, request.request_id)

            for table in all_merged_tables:
                if int(table) not in requests and (table not in links_dict or not self.yt_client.exists(links_dict[table])):
                    logging.info('Deleting table %s because RequestID %s no longer exists', table, table)
                    self.yt_client.remove('{}/{}'.format(merged_dir, table))

        if len(self.yt_client.list(selected_dir)) == 0:
            logging.info('No tables to merge, waiting')
            return

        os.environ['YT_TOKEN'] = self.yt_token
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        run_merge(
            os.path.join(self.get_binary_path(), 'merge'),
            self.Parameters.work_yt_proxy,
            selected_dir,
            merged_dir,
            work_dir,
            YTPUMP_JOBS,
            self.Parameters.log_type,
            self.archive_suffix,
        )

        if self.Parameters.test_run:
            return

        with self.yt_client.Transaction():
            merged_tables = self.yt_client.list(work_dir)
            requests_rows = self.yt_meta_client.lookup_rows(self.meta_path, [{'RequestID': int(t)} for t in merged_tables])
            links_dict = {}
            for row in requests_rows:
                request = DropStatRequest.from_yt_row(row, self.Parameters.log_type)
                if request.params.link_node:
                    links_dict[str(request.request_id)] = request

            for table in merged_tables:
                self.yt_client.remove('{}/{}'.format(selected_dir, table))
                self.yt_client.move(
                    '{}/{}'.format(work_dir, table),
                    '{}/{}'.format(merged_dir, table),
                )
