import datetime
import os

from dateutil.tz import tzlocal

from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess, ProcessLog

from sandbox.projects.yabs.dropstat.base import BaseDropStatTask
from sandbox.projects.yabs.dropstat.base.config import LB_ADDRESS, PRESTABLE_TOPIC_PREFIX, YTPUMP_JOBS


class YabsDropStatSendToLogbroker(BaseDropStatTask):
    '''Send dropstat merged table to logbroker
    '''
    class Parameters(BaseDropStatTask.Parameters):
        description = 'Send dropstat merged table to logbroker'

        with BaseDropStatTask.Parameters.exec_params() as exec_params:
            request_id = parameters.String('RequestID', required=True)

        with parameters.Group('Logbroker') as lb_params:
            tvm_src_id = parameters.Integer(
                'TVM source id',
                default=2018822,
                required=True,
            )
            tvm_dst_id = parameters.Integer(
                'Logbroker TVM id',
                default=2001059,
                required=True,
            )
            tvm_secret = parameters.YavSecret(
                'TVM secret',
                description='Required key: client_secret',
                required=True,
            )

    def on_execute(self):
        with open('secret', 'w') as f:
            f.write(self.Parameters.tvm_secret.data()['client_secret'])

        log_desc = self.log_description
        debug_path = self.work_dir + '/ytpump_debug'
        stats_path = '{}/{}_stats'.format(debug_path, self.Parameters.request_id)
        stderr_path = '{}/{}_stderr'.format(debug_path, self.Parameters.request_id)

        if self.Parameters.production:
            topic = log_desc.lb_topic_name
        else:
            topic = '{}/{}'.format(PRESTABLE_TOPIC_PREFIX, log_desc.lb_topic_name)

        path = '{}/merged/{}'.format(self.work_dir, self.Parameters.request_id)

        bin_path = os.path.join(self.get_binary_path(), 'ytpump')
        os.environ['YT_TOKEN'] = self.yt_token
        os.environ['YT_LOG_LEVEL'] = 'INFO'
        command = [
            bin_path,
            '--cluster', self.Parameters.work_yt_proxy,
            '--input', path,
            '--group-by', log_desc.id_column,
            '--session-by', 'session_id',
            '--ignore-fields', 'session_id,_rest',
            '--logbroker', LB_ADDRESS,
            '--topic', topic,
            '--format', 'tskv',
            '--chunk-size', 8388608,
            '--tvm-src-id', self.Parameters.tvm_src_id,
            '--tvm-dst-id', self.Parameters.tvm_dst_id,
            '--tvm-secret', 'secret',
            '--partition-count', YTPUMP_JOBS,
            '--do-write', 1,
            '--stats-table', stats_path,
            '--stderr-table', stderr_path,
        ]

        with ProcessLog(self, logger='ytpump') as pl:
            subprocess.check_call(map(str, command), stdout=pl.stdout, stderr=pl.stderr)

        ts = datetime.datetime.now(tzlocal())
        ts += datetime.timedelta(days=10)
        for path in (stats_path, stderr_path):
            self.yt_client.set_attribute(path, 'expiration_time', ts.isoformat())
