#!/usr/bin/env python
# -*-coding: utf-8 -*-
# vim: sw=4 ts=4 expandtab ai

import json
import logging

import socket

import requests
import urllib
import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

import pytz
from datetime import datetime, timedelta

import ticket_parser2.api.v1 as tp2


class Calendar():
    def __init__(self, uid, tz, tvm_client_id, tvm_secret, tvm_service_id, oauth_token):
        self.uid = uid
        self.local_tz = pytz.timezone(tz)

        self.tvm_client = tp2.TvmClient(tp2.TvmApiClientSettings(
            self_client_id=tvm_client_id,
            self_secret=tvm_secret,
            dsts={
                "calendar": tvm_service_id,
                "blackbox-prod-yateam": 223,
            },
        ))

        self.oauth_token = oauth_token
        self.blackbox_url = "https://blackbox.yandex-team.ru/blackbox"

        self.session = requests.Session()
        self.session.headers['Content-Type'] = 'application/json'
        self.session.headers['X-Ya-Service-Ticket'] = self.tvm_client.get_service_ticket_for("calendar")
        self.session.headers['X-Ya-User-Ticket'] = self._get_user_ticket()
        self.endpoint = "https://calendar-api.tools.yandex.net/internal"

    def _get_user_ticket(self):
        response = requests.get(
            self.blackbox_url,
            params={
                'method': 'oauth',
                'format': 'json',
                'get_user_ticket': 'true',
                'oauth_token': self.oauth_token,
                'userip': socket.getaddrinfo(socket.gethostname(), 0)[0][4][0],
            },
            headers={
                'X-Ya-Service-Ticket': self.tvm_client.get_service_ticket_for("blackbox-prod-yateam")
            }
        )

        return response.json()['user_ticket']

    def _query_args(self, args):
        return '&'.join("{}={}".format(k, urllib.quote(str(v))) for k, v in args.items())

    def _date_str(self, d):
        return self.local_tz.localize(d).strftime("%Y-%m-%dT%H:%M:%S")

    def _date_str_to_utc_str(self, d):
        d1 = datetime.strptime(d, "%Y-%m-%dT%H:%M:%S")
        d2 = self.local_tz.localize(d1)
        return d2.astimezone(pytz.utc).strftime("%Y-%m-%dT%H:%M:%S")

    def _attendees(self, people):
        email_domain = "yandex-team.ru"
        return ['@'.join([login, email_domain]) for login in people]

    def get_events(self, layer_id, from_date, to_date):
        params = {
            "from": self._date_str(from_date),
            "to": self._date_str(to_date),
            "uid": self.uid,
            "layerId": layer_id,
        }

        url = "{endpoint}/{handler}?{args}".format(endpoint=self.endpoint, handler='get-events', args=self._query_args(params))
        response = self.session.get(url, verify=False).text
        return json.loads(response)["events"]

    def update_event(self, layer_id, event, name, people):
        new_attendees = self._attendees(people)

        for attendee in event["attendees"]:
            if attendee["email"] not in new_attendees:
                new_attendees.append(attendee["email"])

        for resource in event["resources"]:  # add meeting room as email (sic!)
            if resource["email"] not in new_attendees:
                new_attendees.append(resource["email"])

        new_event = {
            "startTs": event["startTs"],
            "endTs": event["endTs"],
            "name": name,
            "attendees": new_attendees,
        }

        params = {
            "uid": self.uid,
            "id": event["id"],
            "layerId": layer_id,
            "sequence": event["sequence"],
            "instanceStartTs": self._date_str_to_utc_str(event["startTs"]),  # calendar api is ugly :(
            "applyToFuture": False,
            "tz": str(self.local_tz),
        }

        url = "{endpoint}/{handler}?{args}".format(endpoint=self.endpoint, handler='update-event', args=self._query_args(params))
        return self.session.post(url, data=json.dumps(new_event), verify=False).text

    def create_event(self, layer_id, start_ts, end_ts, name, description, people):
        params = {
            "uid": self.uid,
        }
        new_event = {
            "type": "user",
            "startTs": self._date_str(start_ts),
            "endTs": self._date_str(end_ts),
            "name": name,
            "location": "",
            "description": description,
            "isAllDay": False,
            "availability": "available",
            "layerId": layer_id,
            "notifications": [],
            "attendees": self._attendees(people),
            "repetition": {},
            "participantsCanInvite": True,
            "participantsCanEdit": True,
            "othersCanView": True
        }

        url = "{endpoint}/{handler}?{args}".format(endpoint=self.endpoint, handler='create-event', args=self._query_args(params))
        return self.session.post(url, data=json.dumps(new_event), verify=False).text


class AbcDutySchedule():
    def __init__(self, service_slug, schedule_slug, wiki, wiki_duty_fields, abc_oauth_token):
        self.service_slug = service_slug
        self.schedule_slug = schedule_slug

        self.wiki = wiki
        self.wiki_duty_fields = wiki_duty_fields

        self.session = requests.Session()
        self.session.headers['Content-Type'] = 'application/json'
        self.session.headers['Authorization'] = 'OAuth {}'.format(abc_oauth_token)
        self.endpoint = "https://abc-back.yandex-team.ru/api/v4"

        self.schedule = self._get_schedule()
        self._disable_recalculation()

    def _get_schedule(self):
        url = "{endpoint}/duty/schedules/?service__slug={slug}&fields=id,slug,name,recalculate".format(endpoint=self.endpoint, slug=self.service_slug)
        r = self.session.get(url)
        logging.debug("_get_schedule request result: {}".format(r.text))
        schedules = r.json()
        for schedule in schedules["results"]:
            if schedule["slug"] == self.schedule_slug:
                return schedule

    def _disable_recalculation(self):
        if self.schedule["recalculate"]:
            url = "{endpoint}/duty/schedules/{schedule_id}/".format(endpoint=self.endpoint, schedule_id=self.schedule["id"])
            data = {"recalculate": False}
            r = self.session.patch(url, data=json.dumps(data))
            logging.debug("_disable_recalculation request result: {}".format(r.text))

    def upload_history(self, duty_type):
        logging.info("loading duty plan from wiki")
        plan = self.wiki.get(self.wiki_duty_fields)
        logging.info("loaded {} duty events".format(len(plan)))

        history = []

        prev_ts = None
        for ts in sorted(plan):
            if prev_ts:
                if duty_type == "first" and type(plan[prev_ts]["duty"]) == list and len(plan[prev_ts]["duty"]) > 0:
                    person = plan[prev_ts]["duty"][0]
                elif duty_type == "second" and type(plan[ts]["duty"]) == list and len(plan[ts]["duty"]) > 0:
                    person = plan[ts]["duty"][0]
                else:
                    person = None

                if person:
                    start_datetime = prev_ts.strftime("%Y-%m-%dT%H:%M:%S")
                    end_datetime = ts.strftime("%Y-%m-%dT%H:%M:%S")

                    history.append({
                        "person": person,
                        "schedule": self.schedule["id"],
                        "start_datetime": start_datetime,
                        "end_datetime": end_datetime,
                    })

            prev_ts = ts

        logging.debug("history: {}".format(json.dumps(history)))

        url = "{endpoint}/duty/schedules/{schedule_id}/shifts/replace/".format(endpoint=self.endpoint, schedule_id=self.schedule["id"])
        r = self.session.post(url, data=json.dumps(history))
        logging.debug("upload_history request result: {}".format(r.text))


class DutyWiki():
    def __init__(self, oauth_token, url, ts_field, ts_format, old_row_field):
        self.oauth_token = oauth_token
        self.ts_field = ts_field
        self.ts_format = ts_format
        self.old_row_field = old_row_field
        self.url = url

        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(oauth_token)
        self.session.headers['Content-Type'] = 'application/json'

        self.body = None

    def get(self, duty_fields):
        if not self.body:
            response = self.session.get(self.url, verify=False).text
            self.body = json.loads(response)
            logging.debug('Got body\n%s', json.dumps(self.body, indent=2))

        field_titles = {}
        for field in self.body["data"]["structure"]["fields"]:
            field_titles[field["name"]] = field["title"]  # name?! omg

        plan = {}
        for raw_row in self.body["data"]["rows"]:
            row = {}
            for col in raw_row:
                f_id = col["__key__"]
                if f_id in field_titles:
                    row[field_titles[f_id]] = col["raw"]

            ts = datetime.strptime(row[self.ts_field], self.ts_format).date()

            for field in duty_fields:
                delta = duty_fields[field]["timedelta_days"]
                change_time = datetime.strptime(duty_fields[field]["duty_change_time"], "%H:%M").time()
                t = datetime.combine(ts + timedelta(days=delta), change_time)

                if row[field]:
                    plan[t] = {
                        "old": row[self.old_row_field],
                        "duty": row[field],
                    }

        return plan


class DutyCalendar():
    def __init__(self, calendar, duty_change_layer_id, duty_event_layer_id, event_tag, wiki, wiki_duty_fields, limit_days=14):
        self.calendar = calendar
        self.duty_change_layer_id = duty_change_layer_id
        self.duty_event_layer_id = duty_event_layer_id
        self.event_tag = event_tag

        self.wiki = wiki
        self.wiki_duty_fields = wiki_duty_fields
        self.limit_days = limit_days

    def update_events(self):
        logging.info("update events for {}".format(self.event_tag))

        logging.info("loading duty plan from wiki")
        plan = self.wiki.get(self.wiki_duty_fields)
        logging.info("loaded {} duty events".format(len(plan)))

        prev_ts = None
        for ts in sorted(plan):
            if not plan[ts]["old"] and ts > datetime.now() and ts < datetime.now() + timedelta(days=self.limit_days):
                prev_duty = plan[prev_ts]["duty"]
                new_duty = plan[ts]["duty"]
                logging.info("preparing events for new duty {} {}, prev duty {} {}".format(ts, ', '.join(new_duty), prev_ts, ', '.join(prev_duty)))

                event_people = set(prev_duty + new_duty)
                event_name = "Передача дежурства {} {} -> {}".format(self.event_tag, ', '.join(prev_duty), ', '.join(new_duty))

                event = None
                events = self.calendar.get_events(layer_id=self.duty_change_layer_id, from_date=ts, to_date=ts)
                for e in events:
                    if self.event_tag in e["description"]:
                        event = e

                if event:
                    logging.info("found duty change event id: {event_id}, sequence: {sequence}, period: {start} - {end}, name: \"{name}\", attendees: {attendees}".format(
                        event_id=event["id"],
                        sequence=event["sequence"],
                        start=event["startTs"],
                        end=event["endTs"],
                        name=event["name"].encode("utf8"),
                        attendees=', '.join([a["login"] for a in event["attendees"]])
                        )
                    )
                    logging.info("update duty change event with new name \"{name}\", append attendees: {attendees}".format(
                        name=event_name,
                        attendees=', '.join(event_people)
                        )
                    )
                    result = self.calendar.update_event(
                        layer_id=self.duty_change_layer_id,
                        event=event,
                        name=event_name,
                        people=event_people
                    )
                    logging.info("result: {}".format(result))

                if self.duty_event_layer_id and not plan[prev_ts]["old"]:
                    event_people = plan[prev_ts]["duty"]
                    event_name = "Дежурство {} {}".format(self.event_tag, ', '.join(event_people))

                    event = None
                    events = self.calendar.get_events(layer_id=self.duty_event_layer_id, from_date=prev_ts, to_date=prev_ts)
                    for e in events:
                        if self.event_tag in e["description"]:
                            event = e

                    if event:
                        logging.info("found duty event id: {event_id}, sequence: {sequence}, period: {start} - {end}, name: \"{name}\", attendees: {attendees}".format(
                            event_id=event["id"],
                            sequence=event["sequence"],
                            start=event["startTs"],
                            end=event["endTs"],
                            name=event["name"].encode("utf8"),
                            attendees=', '.join([a["login"] for a in event["attendees"]])
                            )
                        )
                        logging.info("update duty event with new name \"{name}\", append attendees: {attendees}".format(
                            name=event_name,
                            attendees=', '.join(event_people)
                            )
                        )
                        result = self.calendar.update_event(
                            layer_id=self.duty_event_layer_id,
                            event=event,
                            name=event_name,
                            people=event_people
                        )
                        logging.info("result: {}".format(result))

                    else:
                        logging.info("create duty event {start} - {end} with new name \"{name}\", append attendees: {attendees}".format(
                            start=prev_ts,
                            end=ts,
                            name=event_name,
                            attendees=', '.join(event_people)
                            )
                        )
                        result = self.calendar.create_event(
                            layer_id=self.duty_event_layer_id,
                            start_ts=prev_ts,
                            end_ts=ts,
                            name=event_name,
                            description=self.event_tag,
                            people=event_people,
                        )
                        logging.info("result: {}".format(result))
            prev_ts = ts


if __name__ == "__main__":
    logging.basicConfig(format='%(asctime)s - %(name)s - %(levelname)s - %(message)s', level=logging.INFO)

    calendar = Calendar(
        uid=1120000000051238,  # robot-yabs-rt
        tz='Europe/Moscow',
        tvm_client_id=2018572,
        tvm_secret="<censored>",
        tvm_service_id=2011072,  # calendar y-t prod
        oauth_token="<censored>",
    )

    wiki = DutyWiki(
        oauth_token="<censored>",  # robot-yabs-rt oauth_token
        url="https://wiki-api.yandex-team.ru/_api/frontend/BannernajaKrutilka/duty/DutyPlan/.raw",
        ts_field="Week",
        ts_format="%Y-%m-%d",
        old_row_field=u"История",
    )

    engine_duty = DutyCalendar(
        calendar=calendar,
        duty_change_layer_id=64555,  # Передача дежурства
        duty_event_layer_id=64532,  # Дежурства БК
        event_tag=u"Движок БК",

        wiki=wiki,
        wiki_duty_fields={
            "*Engine1": {
                "timedelta_days": 0,
                "duty_change_time": "16:00",
            },
            "*Engine2": {
                "timedelta_days": 3,
                "duty_change_time": "16:00",
            },
        },
    )

    engine_duty.update_events()
