#!/usr/bin/env python
# -*-coding: utf-8 -*-
# vim: sw=4 ts=4 expandtab ai

import json
import requests
import logging

from time import mktime
from datetime import datetime, date

import yt.wrapper as yt


def get_escalations_log(filters, start_time, end_time):
    escalations = []
    escalation_ids = []

    payload = {
        "filters": filters,
        "only_running": False,
        "page_size": 100,
        "page": 0,
    }

    while payload["page"] < 10:  # get 10 pages max
        data = requests.post("https://juggler-api.search.yandex.net/v2/escalations/get_escalations_log", data=json.dumps(payload), headers={"Content-Type": "application/json"}, verify=False).json()
        if "escalations" in data and data["escalations"]:
            for e in sorted(data["escalations"], key=lambda e: -e['start_time']):
                if e["start_time"] >= start_time and e["start_time"] <= end_time and e["escalation_id"] not in escalation_ids:
                    escalations.append(e)
                    escalation_ids.append(e["escalation_id"])
        else:
            break

        payload["page"] += 1

    return sorted(escalations, key=lambda e: e['start_time'])


def get_escalations_report(juggler_namespaces, juggler_tags, start_date, end_date):
    report = ("DUTY PERIOD {} - {}\n".format(
        start_date.strftime("%Y-%m-%d %H:%M %a"),
        end_date.strftime("%Y-%m-%d %H:%M %a")
    ))

    escalations = get_escalations_log(
        filters=[{
            "tags": juggler_tags,
            "namespace": namespace
        } for namespace in juggler_namespaces],
        start_time=mktime(start_date.timetuple()),
        end_time=mktime(end_date.timetuple()),
    )

    events_stat = {}
    logins_stat = {}
    days_stat = {}

    call_times = []

    day_key = ""

    for e in escalations:

        if e["end_reason"] == "USER_STOPPED":

            host = e["host"]
            service = e["service"]
            login = e["stopped"]["login"]
            start_time = datetime.fromtimestamp(e["start_time"])
            stopped_time = datetime.fromtimestamp(e["stopped"]["time"])
            description = e["stopped"]["description"].replace("\n", " ")

            key = "%s - %s" % (host, service)

            prev_day_key = day_key
            day_key = stopped_time.strftime("%Y-%m-%d %a")
            if prev_day_key != day_key:
                report += "\n"

            call_key = stopped_time.strftime("%Y-%m-%d %H:%M:%S")

            if day_key not in days_stat:
                days_stat[day_key] = {
                    "night_calls": 0,
                    "day_calls": 0,
                    "manual_stops": 0,
                }

            if key not in events_stat:
                events_stat[key] = {
                    "night_calls": 0,
                    "day_calls": 0,
                    "manual_stops": 0,
                    "dates": [],
                }

            if login not in logins_stat:
                logins_stat[login] = {
                    "night_calls": 0,
                    "day_calls": 0,
                    "manual_stops": 0,
                }

            mark = ""
            if description == "Stopped by phone":
                if stopped_time.hour >= 23 or stopped_time.hour <= 8:
                    mark = "NIGHT"
                    events_stat[key]["night_calls"] += 1
                    if call_key not in call_times:
                        days_stat[day_key]["night_calls"] += 1
                        logins_stat[login]["night_calls"] += 1
                        call_times.append(call_key)
                else:
                    events_stat[key]["day_calls"] += 1
                    if call_key not in call_times:
                        days_stat[day_key]["day_calls"] += 1
                        logins_stat[login]["day_calls"] += 1
                        call_times.append(call_key)
            else:
                events_stat[key]["manual_stops"] += 1
                days_stat[day_key]["manual_stops"] += 1
                logins_stat[login]["manual_stops"] += 1

            events_stat[key]["dates"].append(stopped_time)

            report += "{key:70}  {start_time}  {stop_time}  {mark:>5}  {login:>15}  {description}\n".format(
                key=key,
                start_time=start_time.strftime("%Y-%m-%d %H:%M %a"),
                stop_time=stopped_time.strftime("%Y-%m-%d %H:%M %a"),
                mark=mark,
                login=login,
                description=description.encode('utf-8'),
            )

    total_night_calls = 0
    total_day_calls = 0
    total_manual_stops = 0

    report += "\n{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}  {events}\n".format(key="CHECK", total="TOTAL", night="NIGHT", day="DAY", manual="MANUAL", events="EVENTS")
    for key in sorted(events_stat.keys(), key=lambda e: (-(events_stat[e]["night_calls"] + events_stat[e]["day_calls"] + events_stat[e]["manual_stops"]), events_stat[e]["dates"][0])):
        event = events_stat[key]
        total_night_calls += event["night_calls"]
        total_day_calls += event["day_calls"]
        total_manual_stops += event["manual_stops"]
        report += "{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}  {events}\n".format(
            key=key,
            total=event["night_calls"] + event["day_calls"] + event["manual_stops"],
            night=event["night_calls"],
            day=event["day_calls"],
            manual=event["manual_stops"],
            events=', '.join([d.strftime("%a %H:%M") for d in sorted(event["dates"])]),
        )
    report += "{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}\n".format(
        key="TOTAL",
        total=total_night_calls+total_day_calls+total_manual_stops,
        night=total_night_calls,
        day=total_day_calls,
        manual=total_manual_stops,
    )

    total_night_calls = 0
    total_day_calls = 0
    total_manual_stops = 0

    report += "\n{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}\n".format(key="DAY", total="TOTAL", night="NIGHT", day="DAY", manual="MANUAL")
    for key in sorted(days_stat.keys()):
        event = days_stat[key]
        total_night_calls += event["night_calls"]
        total_day_calls += event["day_calls"]
        total_manual_stops += event["manual_stops"]
        report += "{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}\n".format(
            key=key,
            total=event["night_calls"] + event["day_calls"] + event["manual_stops"],
            night=event["night_calls"],
            day=event["day_calls"],
            manual=event["manual_stops"],
        )
    report += "{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}\n".format(
        key="TOTAL",
        total=total_night_calls+total_day_calls+total_manual_stops,
        night=total_night_calls,
        day=total_day_calls,
        manual=total_manual_stops,
    )

    report += "\n{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}\n".format(key="LOGIN", total="TOTAL", night="NIGHT", day="DAY", manual="MANUAL")
    for key in sorted(logins_stat.keys(), key=lambda l: logins_stat[l]["night_calls"] + logins_stat[l]["day_calls"] + logins_stat[l]["manual_stops"], reverse=True):
        login = logins_stat[key]
        report += "{key:70}  {total:>6}  {night:>6}  {day:>6}  {manual:>6}\n".format(
            key=key,
            total=login["night_calls"] + login["day_calls"] + login["manual_stops"],
            night=login["night_calls"],
            day=login["day_calls"],
            manual=login["manual_stops"],
        )

    return report


def get_onduty(service, schedule, session):
    schedule = session.get("https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service__slug={}&schedule__slug={}".format(service, schedule)).json()
    staff_login = schedule[0]["person"]["login"]

    main_telegram_account = None
    private_telegram_account = None

    staff = session.get("https://staff-api.yandex-team.ru/v3/persons?_fields=telegram_accounts&login={}".format(staff_login)).json()
    for account in staff["result"][0]["telegram_accounts"]:
        if not account["private"]:
            main_telegram_account = account["value"]
            break
        else:
            private_telegram_account = account["value"]

    if main_telegram_account:
        telegram_account = main_telegram_account
    elif private_telegram_account:
        telegram_account = private_telegram_account
    else:
        telegram_account = "n/a"

    return staff_login, telegram_account


def send_report_message(abc_token, telegram_token, juggler_namespaces, juggler_tags, abc_duty_service, abc_duty_schedule, telegram_chat_id):
    # get previous duty schedule interval for report
    today = date.today().strftime("%Y-%m-%d")
    schedules = requests.get(
        "https://abc-back.yandex-team.ru/api/v4/duty/shifts/?service__slug={}&schedule__slug={}&date_from={}&date_to={}".format(
            abc_duty_service,
            abc_duty_schedule,
            today,
            today
        ), headers={'Authorization': "OAuth {}".format(abc_token)}
    ).json()

    schedule = sorted(schedules["results"], key=lambda s: s["start"])[0]
    start_datetime = datetime.strptime(schedule["start_datetime"][:19], "%Y-%m-%dT%H:%M:%S")
    end_datetime = datetime.strptime(schedule["end_datetime"][:19], "%Y-%m-%dT%H:%M:%S")

    # generate escalations report
    report = get_escalations_report(juggler_namespaces, juggler_tags, start_datetime, end_datetime)

    # upload it to paste
    payload = {
        'syntax': 'plain',
        'text': report
    }

    paste_server = 'https://paste.yandex-team.ru'
    r = requests.post(paste_server, data=payload, allow_redirects=False, verify=False)
    url = r.headers['Location']
    url = "{}{}/text".format(paste_server, url)

    # send message with report link to chat
    msg = "<a href=\"{url}\">Статистика событий за дежурство {start} - {end}</a> #статистика".format(start=start_datetime.strftime("%Y-%m-%d"), end=end_datetime.strftime("%Y-%m-%d"), url=url)

    url = 'https://api.telegram.org/bot{}/sendMessage'.format(telegram_token)
    params = {
        'text': msg,
        'chat_id': telegram_chat_id,
        'parse_mode': 'html'
    }
    requests.get(url, params=params)


def send_duties_message(
        abc_token,
        telegram_token,
        yt_cluster,
        yt_token,
        yt_path,
        abc_duty_service,
        abc_duty_schedule_first,
        abc_duty_schedule_second,
        abc_supbs_service,
        abc_supbs_engine1_schedule,
        abc_qo_service,
        abc_qo_schedule,
        telegram_chat_id
        ):
    # get duties
    s = requests.Session()
    s.headers['Authorization'] = "OAuth {}".format(abc_token)
    first_login, first_telegram = get_onduty(abc_duty_service, abc_duty_schedule_first, s)

    # send message with dutiles to chat
    if abc_duty_schedule_second:
        msg = u"Дежурные"
    else:
        msg = u"Дежурный"
    msg += u": <a href=\"https://staff.yandex-team.ru/{first_login}\">{first_login}@</a> (@{first_telegram})".format(
        first_login=first_login,
        first_telegram=first_telegram,
    )
    if abc_duty_schedule_second:
        second_login, second_telegram = get_onduty(abc_duty_service, abc_duty_schedule_second, s)
        msg += u", <a href=\"https://staff.yandex-team.ru/{second_login}\">{second_login}@</a> (@{second_telegram})".format(
            second_login=second_login,
            second_telegram=second_telegram,
        )
    if abc_supbs_engine1_schedule:
        supbs_engine_login, supbs_engine_telegram = get_onduty(abc_supbs_service, abc_supbs_engine1_schedule, s)
        msg += u"; Дебаг: <a href=\"https://staff.yandex-team.ru/{supbs_engine_login}\">{supbs_engine_login}@</a> (@{supbs_engine_telegram})".format(
            supbs_engine_login=supbs_engine_login,
            supbs_engine_telegram=supbs_engine_telegram,
        )

    if abc_qo_schedule:
        qo_login, qo_telegram = get_onduty(abc_qo_service, abc_qo_schedule, s)
        msg += u"; Quality Officer: <a href=\"https://staff.yandex-team.ru/{qo_login}\">{qo_login}@</a> (@{qo_telegram})".format(
            qo_login=qo_login,
            qo_telegram=qo_telegram,
        )

    telegram_url = 'https://api.telegram.org/bot{}'.format(telegram_token)
    params = {
        'text': msg,
        'chat_id': telegram_chat_id,
        'parse_mode': 'html'
    }

    logging.debug(msg)
    r = requests.get(telegram_url + "/sendMessage", params=params).json()
    logging.debug(json.dumps(r, indent=2))
    message_id = r['result']['message_id']

    # pin message
    params = {
        'chat_id': telegram_chat_id,
        'message_id': message_id,
        'disable_notification': True
    }

    requests.get(telegram_url + "/pinChatMessage", params=params)

    # get previous message id
    yt.config["proxy"]["url"] = yt_cluster
    yt.config["token"] = yt_token

    try:
        prev_message_id = int(yt.get(yt_path))
    except:
        prev_message_id = None

    # unpin it
    if prev_message_id is not None:
        params = {
            'chat_id': telegram_chat_id,
            'message_id': prev_message_id,
        }
        requests.get(telegram_url + "/unpinChatMessage", params=params)

    # store pinned message id
    yt.set(yt_path, str(message_id))


def send_html_message(telegram_token, telegram_chat_id, message):
    url = 'https://api.telegram.org/bot{}/sendMessage'.format(telegram_token)
    params = {
        'text': message,
        'chat_id': telegram_chat_id,
        'parse_mode': 'html'
    }
    requests.get(url, params=params)
