import time
from enum import Enum

from library.python import resource
from sandbox import sdk2
import sandbox.common.types.task as ctt
import yt.yson as yson

from sandbox.projects.yabs.export_keywords_from_proto_to_yt import main


class Modes(str, Enum):
    ONLY_CREATE_SNAPSHOT = "Only create tables snapshot, do not change any links"
    ONLY_CHANGE_LINKS = "Only change tables links; use it to revert to older tables"
    FULL_MODE = "Create tables snapshot and change links"


class ExportKeywordsFromProtoToYT(sdk2.Task):
    class Requirements(sdk2.Requirements):
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='keyword_exporting_to_yt', weight=1, capacity=1)
            ]
        )

    class Parameters(sdk2.Parameters):  # pylint: disable=too-few-public-methods # noqa
        description = "Export keyword tables to YT"
        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT token",
            default="sec-01d4nnsjv51k0rqtr5fv6qyz8w",
            required=True
        )
        clusters = sdk2.parameters.List(
            "YT clusters (proxies) to export keywords to",
            default=["freud", "hahn", "arnold", "seneca-sas", "seneca-vla", "seneca-man"],
            required=True
        )
        yt_dir_path = sdk2.parameters.String(
            "Path to workdir in ALL clusters",
            default="//home/yabs/dict",
            required=True
        )
        with sdk2.parameters.String("Mode", required=True) as mode:
            mode.values[Modes.FULL_MODE] = mode.Value(Modes.FULL_MODE, default=True)
            mode.values[Modes.ONLY_CREATE_SNAPSHOT] = Modes.ONLY_CREATE_SNAPSHOT
            mode.values[Modes.ONLY_CHANGE_LINKS] = Modes.ONLY_CHANGE_LINKS
        with mode.value["Only change tables links; use it to revert to older tables"]:  # IDK how to fix that, using proper enum value bewaks everything # noqa
            snapshot_id = sdk2.parameters.String("Target suffix for snapshot (all table names end with it).",
                                                 required=True)
        extremely_dangerous_disable_snapshot_verification = sdk2.parameters.Integer(
            "REALLY DANGEROUS! Input current UNIX timestamp (timezone independent! in seconds, with error < 1000) to"
            " disable snapshot verification.",
            default=0,
        )

    def on_execute(self):
        SCHEMA_RESOURCE_NAMES = (
            ("Keyword", "sandbox/projects/yabs/export_keywords_from_proto_to_yt/Keyword-schema.yson"),
            ("KeywordType", 'sandbox/projects/yabs/export_keywords_from_proto_to_yt/KeywordType-schema.yson'),
            ("KeywordDataType", 'sandbox/projects/yabs/export_keywords_from_proto_to_yt/KeywordDataType-schema.yson'),
        )
        schemas_by_table_name = {}
        yt_token = self.Parameters.yt_token.data().get("yt_token", None) or self.Parameters.yt_token.data()["token"]
        for table_name, resource_name in SCHEMA_RESOURCE_NAMES:
            schema_resource_data = resource.find(resource_name)
            schema = yson.loads(schema_resource_data)
            schemas_by_table_name[table_name] = schema
        should_create_new_snapshot = self.Parameters.mode in (Modes.ONLY_CREATE_SNAPSHOT, Modes.FULL_MODE)
        should_change_links = self.Parameters.mode in (Modes.ONLY_CHANGE_LINKS, Modes.FULL_MODE)
        if abs(self.Parameters.extremely_dangerous_disable_snapshot_verification - time.time()) < 1000 and self.Parameters.snapshot_id: # noqa
            main.EXTREMELY_DANGEROUS_DISABLE_SNAPSHOT_VERIFICATION = True
        main.main(
            schemas_by_table_name,
            clusters=self.Parameters.clusters,
            yt_dir_path=self.Parameters.yt_dir_path,
            yt_token=yt_token,
            snapshot_id=self.Parameters.snapshot_id,
            should_create_new_snapshot=should_create_new_snapshot,
            should_change_links=should_change_links,
            sandbox_task=self
        )
