import logging

from sandbox.sdk2 import (
    ResourceData,
    Vault,
    parameters,
)
from sandbox.common.types.resource import State
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.infra.mdb_clickhouse_apply_config.resources import (
    MdbClickhouseDictsConfig,
    MdbClickhouseUsersConfig,
    MdbClickhouseServerConfig,
)


class MdbClickhouseApplyConfig(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        description = "Task for applying new configuration to Clickhouse cluster in MDB(Yandex Cloud)"

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            release_version = parameters.String(
                "Release version",
                default="stable"
            )
            resource_attrs = parameters.Dict(
                "Filter resource by",
                default={"name": "MdbClickhouseApplyConfig"},
            )

        with parameters.Group("Resources") as resource_params:
            dicts_config = parameters.Resource(
                "Dicts config resource",
                resource_type=MdbClickhouseDictsConfig,
                state=State.READY,
            )
            users_config = parameters.Resource(
                "Users config resource",
                resource_type=MdbClickhouseUsersConfig,
                state=State.READY,
            )
            server_config = parameters.Resource(
                "Server config resource",
                resource_type=MdbClickhouseServerConfig,
                state=State.READY,
            )

        with parameters.Group("Execution parameters") as exec_params:
            cluster_id = parameters.String(
                "MDB cluster ID",
                required=True,
            )
            mdb_token = parameters.String(
                "Vault MDB token",
                required=True,
            )
            yav_token = parameters.String(
                "Vault YAV token",
                required=True,
            )
            retries = parameters.Integer(
                "Retries",
                default=3,
                required=True,
            )

    @staticmethod
    def _get_config_path(config_resource):
        if config_resource:
            return str(ResourceData(config_resource).path)
        else:
            return None

    def on_execute(self):
        from yabs.stat.infra.clickhouse.lib import (
            SecretResolver,
            get_iam_token,
            ClusterClient,
        )
        from library.python.vault_client.instances import Production as VaultClient

        vault_client = VaultClient(
            authorization=Vault.data(self.owner, self.Parameters.yav_token),
            rsa_auth=False,
            decode_files=True,
        )
        secret_resolver = SecretResolver(vault_client)
        iam_token = get_iam_token(
            Vault.data(self.owner, self.Parameters.mdb_token)
        )
        cluster = ClusterClient(iam_token, secret_resolver, cluster_id=self.Parameters.cluster_id)

        server = self._get_config_path(self.Parameters.server_config)
        dicts = self._get_config_path(self.Parameters.dicts_config)
        users = self._get_config_path(self.Parameters.users_config)

        if not (server or users or dicts):
            return

        for retry in range(self.Parameters.retries):
            try:
                cluster.apply_configs(server, users, dicts)
                break
            except Exception as e:
                logging.warn('Failed to apply: %s', e)
                if retry == self.Parameters.retries - 1:
                    raise
