import logging

from sandbox import sdk2
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper

logger = logging.getLogger(__name__)


class ReportReleaseResources(sdk2.Task):
    __st_helper = None
    __component_info = None

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        release_number = sdk2.parameters.Integer('Release number')
        component_resources = sdk2.parameters.Dict('Component resources (resource_name:resource_id)')

        with sdk2.parameters.Group('Report settings') as report_settings:
            message_template = sdk2.parameters.String(
                'Message template (Ex: "resource link: {resource_name.url} task id: {resource_name.task.id}")',
                default='')
            post_st_comment = sdk2.parameters.Bool('Post to startrek release ticket', default=True)
            send_tg_message = sdk2.parameters.Bool('Send to component-defined telegram chat', default=False)

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def st_helper(self):
        if self.__st_helper is None:
            token = rm_sec.get_rm_token(self)
            self.__st_helper = STHelper(token)
        return self.__st_helper

    @property
    def component_info(self):
        if self.__component_info is None:
            self.__component_info = rmc.COMPONENTS[self.Parameters.component_name]()
        return self.__component_info

    def on_execute(self):
        resources = {resource_name: sdk2.Resource[self.Parameters.component_resources[resource_name]] for resource_name
                     in self.Parameters.component_resources}
        message = self.Parameters.message_template.format(**resources)
        logger.debug(message)

        if self.Parameters.post_st_comment:
            self.st_helper.comment(self.Parameters.release_number, message, self.component_info)

        if self.Parameters.send_tg_message:
            notify_parameters = [rm_const.RM_USERS.get(chat) for chat in self.component_info.notify_cfg__tg__chats]
            if len(notify_parameters) == 0:
                logger.error('No {} user found in RM_USERS config, cannot send message'.format(
                    self.component_info.notify_cfg__tg__chats))
            else:
                chat_ids = [chat.telegram for chat in notify_parameters]
                rm_notify.send_tm_message(self, message, chat_ids)
