# coding=utf-8
from __future__ import unicode_literals

import re
import tarfile
import textwrap

import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.projects.yabs.infra.solomon_sender import SolomonSenderMixIn


def ansi_to_html(ansi, dark=True):
    css = "\n".join([
        '.ansi_foreground {{ color: {}; }}'.format(('#000000', '#FFFFFF')[dark]),
        '.ansi_background {{ background-color: {}; white-space: pre-wrap; }}'.format(('#FFFFFF', '#000000')[dark]),
        '.ansi1 { font-weight: bold; }',
        '.ansi3 { font-weight: italic; }',
        '.ansi4 { text-decoration: underline; }',
        '.ansi9 { text-decoration: line-through; }',
        '.ansi30 { color: #000000; }',
        '.ansi31 { color: #FF0000; }',
        '.ansi32 { color: #00FF00; }',
        '.ansi33 { color: #FFFF00; }',
        '.ansi34 { color: #0000FF; }',
        '.ansi35 { color: #FF00FF; }',
        '.ansi36 { color: #00FFFF; }',
        '.ansi37 { color: #FFFFFF; }',
        '.ansi40 { background-color: #000000; }',
        '.ansi41 { background-color: #FF0000; }',
        '.ansi42 { background-color: #00FF00; }',
        '.ansi43 { background-color: #FFFF00; }',
        '.ansi44 { background-color: #0000FF; }',
        '.ansi45 { background-color: #FF00FF; }',
        '.ansi46 { background-color: #00FFFF; }',
        '.ansi47 { background-color: #FFFFFF; }',
        '.ansi90 { color: #000000; }',
        '.ansi91 { color: #FF0000; }',
        '.ansi92 { color: #00FF00; }',
        '.ansi93 { color: #FFFF00; }',
        '.ansi94 { color: #0000FF; }',
        '.ansi95 { color: #FF00FF; }',
        '.ansi96 { color: #00FFFF; }',
        '.ansi97 { color: #FFFFFF; }',
    ])
    supported_codes = [1, 3, 4, 9, 30, 31, 32, 33, 34, 35, 36, 37, 40, 41, 42, 43, 44, 45, 46, 47, 90, 91, 92, 93,
                       94, 95, 96, 97]

    def block_to_html(text):
        match = re.match(r'^\[(?P<code>\d+(?:;\d+)*)?(?P<command>[Am])', text)
        if match is None:
            return text
        text = text[match.end():]
        if match.group('code') is None:
            return text
        classes = []
        for code in match.group('code').split(';'):
            if int(code) in supported_codes:
                classes.append('ansi{0}'.format(code))
        if classes:
            text = '<span class="{0}">{1}</span>'.format(' '.join(classes), text)
        return text

    text = ''.join([block_to_html(block) for block in ansi.split('\x1B')])
    template = """
    <html>
    <head><style>{0}</style></head>
        <body>
            <pre class="ansi_foreground ansi_background">{1}</pre>
        </body>
    </html>
    """
    return template.format(css, text)


class RunBinaryResource(SolomonSenderMixIn):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(SolomonSenderMixIn.Parameters):
        with sdk2.parameters.Group('Resource:') as bin:
            resource_by_id = sdk2.parameters.Bool('resource_by_id', default=False)
            unpack = sdk2.parameters.Bool('unpack resource', default=False)
            with resource_by_id.value[False]:
                resource_type = sdk2.parameters.String('resource_type', required=True)
                resource_attrs = sdk2.parameters.Dict('resource_attrs')
            with resource_by_id.value[True]:
                resource = sdk2.parameters.Resource('resource', required=True)

        with sdk2.parameters.Group('Run:') as run:
            cmd = sdk2.parameters.String(
                'cmd',
                description=textwrap.dedent('''\
                    Command line template using {resource}
                    Ex. for package: {resource}/bin --config {resource}/conf.json --v
                '''),
                required=True,
                default_value='{resource}',
            )
            env_vars = sdk2.parameters.Dict(
                'env_vars',
                description=textwrap.dedent('''\
                    Environment variables (e.g. key value).
                    May be used with Yav Secret: sec-xxx:key
                ''')
            )

    def attach_pretty_log(self, ansi_out):
        resource_log = sdk2.service_resources.TaskCustomLogs(self, 'cmd output', "stdout_stderr.html")
        resource_log_data = sdk2.ResourceData(resource_log)
        resource_log_data.path.write_text(ansi_to_html(ansi_out))
        resource_log_data.ready()
        self.set_info("There are <a href={}>cmd log</a>".format(resource_log.http_proxy), do_escape=False)

    def on_execute(self):
        env = {}
        for k, v in self.Parameters.env_vars.items():
            if v.startswith('sec-'):
                ver, var = v.split(':', 1)
                v = sdk2.yav.Secret(ver).data()[var]
            elif v.startswith('vault:'):
                _, owner, name = v.split(':', 2)
                v = sdk2.Vault.data(owner, name)
            env[k] = v

        if self.Parameters.resource_by_id:
            binary = self.Parameters.resource
        else:
            # using latest resource (sorted by id) with required attrs
            binary = sdk2.Resource.find(type=self.Parameters.resource_type, state=ctr.State.READY,
                                        attrs=self.Parameters.resource_attrs).order(-sdk2.Resource.id).first()
            self.set_info(
                'There are <a href="https://sandbox.yandex-team.ru/resource/{}">resource</a>'.format(binary.id),
                do_escape=False)
        binary_path = sdk2.ResourceData(binary).path.as_posix()
        if self.Parameters.unpack:
            work_dir = str(self.path())
            tar_path = str(sdk2.ResourceData(binary).path)
            with tarfile.open(tar_path) as tar:
                tar.extractall(path=work_dir)
            binary_path = str(self.path())

        try:
            ansi_out = sdk2.helpers.subprocess.check_output(
                self.Parameters.cmd.format(resource=binary_path),
                shell=True,
                env=env,
                stderr=sdk2.helpers.subprocess.STDOUT)
            self.attach_pretty_log(ansi_out)
        except sdk2.helpers.subprocess.CalledProcessError as error:
            ansi_out = error.output
            self.attach_pretty_log(ansi_out)
            message = '{} failed, with status code {}'.format(error.cmd, error.returncode)
            raise RuntimeError(message)
