import time

from sandbox import sdk2
from sandbox.projects.common.solomon import push_to_solomon_v2


class SolomonSenderMixIn(sdk2.Task):
    task_time = int(time.time())

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Solomon parameters', collapse=True) as solomon_parameters:
            solomon_is_enabled = sdk2.parameters.Bool(
                'Enable monitoring', default=False)
            with solomon_is_enabled.value['true']:
                project = sdk2.parameters.String(
                    'Project', required=True, default='yabs'
                )
                cluster = sdk2.parameters.String(
                    'Cluster', required=True, default='yabs'
                )
                service = sdk2.parameters.String(
                    'Service', required=True, default='monitorings_sandbox'
                )
                task = sdk2.parameters.String(
                    'Task', required=True
                )
                token = sdk2.parameters.String(
                    'Solomon Token', required=True, description='Yav secret with token (sec-xxx:key)', default='sec-01dn262krxeace953ng1awtw4x:robot-yabs-bases_solomon-oauth'
                )

    @property
    def token(self):
        v = self.Parameters.token
        if v.startswith('sec-'):
            ver, var = v.split(':', 1)
            v = sdk2.yav.Secret(ver).data()[var]
        return v

    @property
    def solomon_is_enabled(self):
        return self.Parameters.solomon_is_enabled

    @property
    def shard_labels(self):
        return {
            'project': self.Parameters.project,
            'cluster': self.Parameters.cluster,
            'service': self.Parameters.service
        }

    def solomon_sensor(self, status, ts, value=0):
        return {
            'labels': {
                'sensor': 'task_status',
                'task': self.Parameters.task,
                'status': status
            },
            'ts': ts,
            'value': value
        }

    def prepare_sensors(self, status):
        sensors = {
            'success': self.solomon_sensor('success', self.task_time),
            'break': self.solomon_sensor('break', self.task_time),
            'failure': self.solomon_sensor('failure', self.task_time),
            'timeout': self.solomon_sensor('timeout', self.task_time),
        }
        sensors[status]['value'] = 1
        return [
            sensors['success'],
            sensors['break'],
            sensors['failure'],
            sensors['timeout'],
        ]

    def push_sensors(self, sensors):
        push_to_solomon_v2(self.token, self.shard_labels, sensors, common_labels=())

    def on_break(self, prev_status, status):
        if self.solomon_is_enabled:
            self.push_sensors(self.prepare_sensors('break'))
        super(SolomonSenderMixIn, self).on_break(prev_status, status)

    def on_success(self, prev_status):
        if self.solomon_is_enabled:
            self.push_sensors(self.prepare_sensors('success'))
        super(SolomonSenderMixIn, self).on_success(prev_status)

    def on_failure(self, prev_status):
        if self.solomon_is_enabled:
            self.push_sensors(self.prepare_sensors('failure'))
        super(SolomonSenderMixIn, self).on_failure(prev_status)

    def on_timeout(self, prev_status):
        if self.solomon_is_enabled:
            self.push_sensors(self.prepare_sensors('timeout'))
        super(SolomonSenderMixIn, self).on_timeout(prev_status)
