import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.apphost.util import wait_ping


class MobileAdsSdkSUT(ModuleBase):
    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._active = False

        self._ports = dict()
        self._ports['mobile_ads_sdk_port'] = self.adapter.get_mobile_ads_sdk_port()

        self._unpack_mobile_ads_sdk()
        self._unpack_jdk()

        self._disable_monitoring_and_change_port()

    def _unpack_jdk(self):
        self._jdk_unpacked_path = os.path.join(self.adapter.get_work_dir(), 'jdk')
        os.makedirs(self._jdk_unpacked_path, mode=0o777)
        self._java_path = os.path.join(self._jdk_unpacked_path, 'bin', 'java')

        logging.info('Unpacking jdk resource to {}'.format(os.path.join(self.adapter.get_work_dir(), 'jdk')))
        with sdk2.helpers.ProcessLog(self.adapter.task_instance, 'jdk.unpack') as process_log:
            subprocess.Popen(
                ['tar', '-xvf', self.adapter.get_jdk_resource_path(), '-C', os.path.join(self.adapter.get_work_dir(), 'jdk')],
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            ).wait()

    def _unpack_mobile_ads_sdk(self):
        self._mobile_ads_sdk_unpacked_path = os.path.join(self.adapter.get_work_dir(), 'mobile_ads_sdk')
        os.makedirs(self._mobile_ads_sdk_unpacked_path, mode=0o777)

        logging.info('Unpacking mobile ads sdk resource to {}'.format(self._mobile_ads_sdk_unpacked_path))
        with sdk2.helpers.ProcessLog(self.adapter.task_instance, 'mobile_ads_sdk.unpack') as process_log:
            subprocess.Popen(
                ['tar', '-xvf', self.adapter.get_mobile_ads_sdk_resource_path(), '-C', self._mobile_ads_sdk_unpacked_path],
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            ).wait()

    def _disable_monitoring_and_change_port(self):
        config_path = os.path.join(self._mobile_ads_sdk_unpacked_path, 'app', 'node.properties')
        with open(config_path, 'r') as f:
            config = f.read()

        config = config.replace('system.io.server.service.port=80', 'system.io.server.service.port={}'.format(self._ports['mobile_ads_sdk_port']))
        config += 'monitoring.disable=true\n'
        config += 'custom.debug_output=true'

        logging.info("Mobile ads sdk config:\n" + config)

        with open(config_path, 'w') as f:
            f.write(config)

    def _run_mobile_ads_sdk(self):
        cmd = [
            os.path.join(self._mobile_ads_sdk_unpacked_path, 'app', 'start.sh'),
            '--java_path',
            self._java_path,
            '--corecount',
            str(self.adapter.get_core_count()),
            '--app_path',
            os.path.join(self._mobile_ads_sdk_unpacked_path, 'app'),
        ]
        if hasattr(self.adapter, 'is_debug') and self.adapter.is_debug:
            cmd.append("-d")
        logging.info('Running mobile ads sdk with command: {}'.format(cmd))

        self._mobile_ads_sdk_process_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'mobile_ads_sdk.run')
        self._mobile_ads_sdk_process_log_context.__enter__()
        logging.info("MobileAdsSdkLog stdout: {}, stderr: {}".format(self._mobile_ads_sdk_process_log_context.stdout, self._mobile_ads_sdk_process_log_context.stderr))
        self._mobile_ads_sdk_process = subprocess.Popen(
            cmd, stdout=self._mobile_ads_sdk_process_log_context.stdout, stderr=self._mobile_ads_sdk_process_log_context.stderr
        )

    def _stop_mobile_ads_sdk(self, *args):
        self._mobile_ads_sdk_process.kill()
        self._mobile_ads_sdk_process.communicate()
        self._mobile_ads_sdk_process_log_context.__exit__(*args)
        self._mobile_ads_sdk_process_log_context = None

    def __enter__(self):
        self._run_mobile_ads_sdk()

        self._active = True
        wait_ping("http://localhost:{}/sdk_router".format(self._ports['mobile_ads_sdk_port']))
        wait_ping("http://localhost:{}/sdk_renderer".format(self._ports['mobile_ads_sdk_port']))
        wait_ping("http://localhost:{}/sdk_demo".format(self._ports['mobile_ads_sdk_port']))

        return self

    def __exit__(self, *args):
        logging.info('Exiting mobile ads sdk')
        self._stop_mobile_ads_sdk(*args)
        self._active = False

    def get_port(self):
        if self._active:
            return self._ports['mobile_ads_sdk_port']
        else:
            return None

    def __del__(self):
        pass
