import logging
import json
import re

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from requests.exceptions import RequestException
from sandbox import sdk2


logger = logging.getLogger(__name__)

JUGGLER_PUSH_API_URL = 'http://juggler-push.search.yandex.net/events'
BIGB_ORACLE_VALIDATE_CONFIG_URL = 'http://validatorforab-bigboracle.n.yandex-team.ru/validate_config_info'
PREFIX_AB_EXPERIMENT_URL = 'https://ab.yandex-team.ru'
RETRY_IN_ORACLE = 3


def requests_retry_session(retries=3):
    session = requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('https://', adapter)
    return session


def send_raw_event_to_juggler(raw_event):
    try:
        resp = requests.post(JUGGLER_PUSH_API_URL, json={
            'source': 'sandbox',
            'events': [raw_event]
        })
        resp.raise_for_status()
        resp_json = resp.json()
    except (RequestException, ValueError) as e:
        logger.exception(e)
        return None
    return resp_json


def get_status_from_oracle():
    try:
        resp = requests_retry_session().post(BIGB_ORACLE_VALIDATE_CONFIG_URL)
        resp.raise_for_status()
        resp_json = resp.json()
    except (RequestException, ValueError) as e:
        logger.exception(e)
        return {"status": "request_bad", "message": "Bad request to oracle: \n" + str(e)}
    return resp_json


def make_url_to_conflict_testids(message):
    result = ''
    pair_testid_conflicts = re.findall(r'\[\d+, \d+\]', message)
    for pair_testid_conflict in pair_testid_conflicts:
        testids = re.findall(r'\d+', pair_testid_conflict)
        result += PREFIX_AB_EXPERIMENT_URL + '/testid/' + testids[0] + ' - ' + \
                  PREFIX_AB_EXPERIMENT_URL + '/testid/' + testids[1] + '; '
    return result


def make_conflict_message(response):
    msg = 'View full conflict in config: ' + BIGB_ORACLE_VALIDATE_CONFIG_URL + '\n' + response["message"] + ' Config has Conflict' + '\n'
    for conflict in response["conflict"]:
        for conflict_ticket in conflict["conflict_ticket"]:
            msg += conflict["ticket"] + "  -  " + conflict_ticket["ticket"] + ": "
            msg += make_url_to_conflict_testids(conflict_ticket["message"]) + '\n'
    return msg


class ABExperimentValidateConfigInfo(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        juggler_host = sdk2.parameters.String('Juggler host', required=True)
        juggler_service = sdk2.parameters.String('Juggler service', required=True)
        juggler_tags = sdk2.parameters.List('List of juggler tags')

    def on_execute(self):

        response = {}
        for i in range(RETRY_IN_ORACLE):
            response = get_status_from_oracle()
            if response["status"] != "request_bad":
                break

        juggler_response = self.send_status_to_juggler(response)
        if juggler_response is None:
            logger.error('Failed to send Juggler event')

    def send_status_to_juggler(self, response):
        juggler_event = {
            'host': self.Parameters.juggler_host,
            'service': self.Parameters.juggler_service,
            'tags': self.Parameters.juggler_tags,
        }
        if response["status"] == "conflict":
            juggler_event.update({
                'status': 'CRIT',
                'description': make_conflict_message(response)
            })
        elif response["status"] == "OK":
            juggler_event.update({
                'status': 'OK',
                'description': response["message"]
            })
        elif response["status"] == "request_bad":
            juggler_event.update({
                'status': 'CRIT',
                'description': response["message"]
            })

        logger.debug(
            'Juggler event: %s',
            json.dumps(juggler_event, indent=4, sort_keys=True)
        )
        return send_raw_event_to_juggler(juggler_event)
