import json
import logging


logger = logging.getLogger(__name__)


class RunType(object):
    PRECOMMIT = "precommit"
    COMMIT = "commit"
    ONESHOT = "oneshot"
    EXPERIMENT_SWITCHER = "experiment switcher"
    AB_EXPERIMENT_TEST = "AB-experiment test"
    UNKNOWN = "unknown"


class AggregationKey(object):
    class Sandbox(object):
        BIN_DB_LIST = "bin_db_list"
        TASK_TYPE = "task_type"
        RUN_TYPE = "run_type"

    class Yt(object):
        OPERATION_STATE = "operation_state"
        CLUSTER = "yt_cluster"
        POOL = "pool"

        class Annotations(object):
            SERVANT_NAME = "servant_name"
            CONTENT_SYSTEM_KEY = "content_system_key"
            TAGS = "tags"


def get_aggr_key(op, aggr_by, tasks_info, task_id_by_operation_id):
    key = []
    if any([
        aggr_column in aggr_by
        for aggr_column in AggregationKey.Sandbox.__dict__.values()
    ]):
        task_info = tasks_info.get(task_id_by_operation_id[op["operation_id"]])
        if not task_info:
            logger.info("No task info for operation %s", op["operation_id"])
    if set(aggr_by) & {AggregationKey.Yt.POOL, AggregationKey.Yt.Annotations.SERVANT_NAME, AggregationKey.Yt.Annotations.CONTENT_SYSTEM_KEY}:
        spec = json.loads(op.get("spec", "{}"))

    for aggr_key in aggr_by:
        if aggr_key == AggregationKey.Sandbox.TASK_TYPE:
            if task_info:
                key.append(task_info.task_type)
            else:
                key.append("unknown_task_type")
        elif aggr_key == AggregationKey.Sandbox.RUN_TYPE:
            if task_info:
                key.append(task_info.run_type)
            else:
                key.append(RunType.UNKNOWN)
        elif aggr_key == AggregationKey.Sandbox.BIN_DB_LIST:
            if task_info:
                key.append(task_info.bin_db_list)
            else:
                key.append("unknown_bin_db_list")

        elif aggr_key == AggregationKey.Yt.OPERATION_STATE:
            key.append(op["event_type"])
        elif aggr_key == AggregationKey.Yt.CLUSTER:
            key.append(op["cluster_name"])
        elif aggr_key == AggregationKey.Yt.POOL:
            key.append(spec.get("pool", "unknown_pool"))

        elif aggr_key == AggregationKey.Yt.Annotations.SERVANT_NAME:
            annotations = spec.get("annotations", {})
            key.append(annotations.get("servant_name", "unknown_servant"))
        elif aggr_key == AggregationKey.Yt.Annotations.CONTENT_SYSTEM_KEY:
            annotations = spec.get("annotations", {})
            key.append(annotations.get("content_system_key", "unknown_cs_key"))
        else:
            key.append("Unknown")
    return tuple(key)


def get_bucket(buckets, value):
    """
    >>> get_bucket([1], 2)
    >>> get_bucket([1, 2], 1.5)
    2
    >>> get_bucket([1, 2], 0.5)
    1
    >>> get_bucket([1, 2, 3], 0.5)
    1
    """
    left = 0
    right = len(buckets) - 1
    if not buckets or buckets[right] < value:
        return None

    while abs(left - right) > 1:
        middle = (left + right) / 2
        if buckets[middle] < value:
            left = middle
        else:
            right = middle
    return buckets[left] if value < buckets[left] else buckets[right]
