import posixpath
import logging
from collections import namedtuple
from datetime import datetime, timedelta


KB = 1024
MB = 1024 * KB
GB = 1024 * MB
TIME_FORMAT = "%Y-%m-%dT%H:%M:%S.%fZ"

logger = logging.getLogger(__name__)
OperationInfo = namedtuple("OperationInfo", ["id", "operation_state", "title", "start_time", "finish_time", "spec", "progress"])


class OperationState(object):
    COMPLETED = "completed"
    FAILED = "failed"
    ABORTED = "aborted"


class FilterEventsMapper(object):
    def __init__(self, from_time=None, to_time=None, users=(), cluster_names=(), event_types=("operation_completed", "operation_failed")):
        self.from_time = from_time
        self.to_time = to_time
        self.users = users
        self.cluster_names = cluster_names
        self.event_types = event_types

    def __call__(self, row):
        if row["event_type"] in self.event_types:
            finish_time = datetime.strptime(row["finish_time"], TIME_FORMAT)
            if all([
                not self.from_time or self.from_time < finish_time,
                not self.to_time or finish_time < self.to_time,
                not self.users or row["authenticated_user"] in self.users,
                not self.cluster_names or row["cluster_name"] in self.cluster_names,
            ]):
                yield row


def init_tmp_table(yt_client, tmp_dir, task_id, tmp_table_ttl_s=2 * 60 * 60):
    tmp_table = posixpath.join(tmp_dir, str(task_id))
    yt_client.create("table", tmp_table, recursive=True, ignore_existing=True)
    yt_client.set_attribute(tmp_table, "expiration_time", (datetime.utcnow() + timedelta(seconds=tmp_table_ttl_s)).isoformat())

    return tmp_table


def schedule_operation(
    yt_client,
    tmp_table,
    log_tables=(),
    operation_states=(OperationState.COMPLETED, ),
    **kwargs
):
    from yt.wrapper import JsonFormat

    return yt_client.run_map(
        FilterEventsMapper(event_types=list("operation_{}".format(state) for state in operation_states), **kwargs),
        log_tables,
        tmp_table,
        format=JsonFormat(),
        spec={
            "data_weight_per_job": 2 * GB,
        },
        sync=False,
    )


def iter_operations_info(yt_client, op, tmp_table):
    from yt.wrapper import YtOperationFailedError

    try:
        op.wait()
    except YtOperationFailedError as err:
        for host_stderrs in op.get_stderrs():
            logging.error("\n".join(["--- STDERR FROM {}:\n{}\n------------------".format(host, stderr) for host, stderr in host_stderrs.items()]))
        logging.error(err)
        raise
    for row in yt_client.read_table(tmp_table, format="json"):
        yield row
