import logging
import re
from collections import namedtuple

from .aggregation import AggregationKey, RunType


TASK_ID_PATTERN = re.compile(r".*\/(\d{9})\/.*")
TaskInfo = namedtuple("TaskInfo", ("task_type", "run_type", "bin_db_list"))


logger = logging.getLogger(__name__)


def get_task_id_from_operation_spec(operation_spec, operation_id):
    table_paths = set()
    tables = filter(None, operation_spec.get("output_table_paths", []) + [operation_spec.get("output_table_path")])
    for table in tables:
        if table is None:
            continue
        if isinstance(table, dict):
            table_paths.add(table["$value"])
        elif isinstance(table, basestring):
            table_paths.add(table)
        else:
            pass
    task_ids = set()
    for table in table_paths:
        try:
            task_ids.add(int(TASK_ID_PATTERN.findall(table)[0]))
        except (KeyError, IndexError, TypeError):
            pass
    if len(task_ids) == 1:
        return list(task_ids)[0]
    elif len(task_ids) > 1:
        logger.warning("Got %d task_ids %s from %s", len(task_ids), task_ids, operation_id)
        return list(task_ids)[0]
    else:
        logger.warning("Got no task_ids from %s (%s)", operation_id, operation_spec.get("title", None))
        return None


def get_parameter_from_context(task_id, parameter_name, sb_client, default=None):
    return sb_client.task[task_id].context.read().get(parameter_name, default)


def get_tasks_info(task_ids, sb_client, limit=50, aggr_by=()):
    task_slices = [task_ids[i:i + limit] for i in range(0, len(task_ids), limit)]
    result = {}
    for task_slice in task_slices:
        tasks = sb_client.task.read(id=task_slice, limit=limit, children=True, hidden=True)["items"]
        for task in tasks:
            if AggregationKey.Sandbox.BIN_DB_LIST in aggr_by:
                bin_db_list_str = task["input_parameters"].get("bin_db_list") or get_parameter_from_context(task["id"], "bin_db_list", sb_client, default='')
            else:
                bin_db_list_str = 'ANY'
            if AggregationKey.Sandbox.RUN_TYPE in aggr_by:
                run_type = get_run_type(task, sb_client)
            else:
                run_type = "ANY"

            result[task["id"]] = TaskInfo(
                task["type"],
                run_type,
                tuple(bin_db_list_str.split(" ")),
            )
    if len(result) < len(task_ids):
        logger.warning("Cannot get info for following tasks: %s", set(task_ids) - set(result))

    return result


def get_base_task(task_id, sb_client):
    parent_task = sb_client.task.read(id=task_id, limit=1, children=True, hidden=True)["items"][0]
    if parent_task["parent"]:
        return get_base_task(parent_task["parent"]["id"], sb_client)
    return parent_task


def get_run_type(task_info, sb_client):
    for tag in task_info["tags"]:
        if tag == "TESTENV-PRECOMMIT-CHECK":
            return RunType.PRECOMMIT
        elif tag == "TESTENV-COMMIT-CHECK":
            return RunType.COMMIT

    if task_info["parent"]:
        base_task = get_base_task(task_info["parent"]["id"], sb_client)
        for tag in base_task["tags"]:
            if tag == "TESTENV-PRECOMMIT-CHECK":
                return RunType.PRECOMMIT
            elif tag == "TESTENV-COMMIT-CHECK":
                return RunType.COMMIT

        if base_task["type"] == "YABS_SERVER_ONE_SHOT":
            return RunType.ONESHOT
        elif base_task["type"] == "YABS_SERVER_METER":
            return RunType.EXPERIMENT_SWITCHER
        elif base_task["type"] == "YABS_SERVER_AB_EXPERIMENT_TEST":
            return RunType.AB_EXPERIMENT_TEST
    logger.warning("Cannot detect run_type for task %s", task_info["id"])
    return RunType.UNKNOWN
