import logging
import json
from datetime import datetime
from collections import defaultdict, namedtuple

from .operations import TIME_FORMAT


JobMetricKey = namedtuple("JobMetricKey", ("sensor", "labels"))


logger = logging.getLogger(__name__)


class Metric(object):
    def __init__(self, _min=None, _max=None, _sum=None, _count=0):
        self.min = _min
        self.max = _max
        self.sum = _sum
        self.count = _count

    def __add__(self, other):
        _max = max(self.max, other.max) if self.max is not None else other.max
        _min = min(self.min, other.min) if self.min is not None else other.min
        _sum = (self.sum + other.sum) if self.sum is not None else other.sum
        _count = self.count + other.count
        return Metric(_min, _max, _sum, _count)

    def __mul__(self, number):
        if isinstance(number, (int, float)):
            _max = self.max if self.max is None else self.max * number
            _min = self.min if self.min is None else self.min * number
            _sum = self.sum if self.sum is None else self.sum * number
            _count = self.count
            return Metric(_min, _max, _sum, _count)
        raise ValueError("Cannot multiply {} and {}".format(type(self), type(number)))

    def __str__(self):
        return "{}/{}/{} ({})".format(self.min, (self.sum or 0) / (self.count or 1), self.max, self.count)


def get_jobs_metric(job_metric_dict):
    metrics = defaultdict(Metric)
    for job_status, status_metric in job_metric_dict.items():
        for job_type, job_type_metric in status_metric.items():
            metrics[job_type] += Metric(job_type_metric['min'], job_type_metric['max'], job_type_metric['sum'], job_type_metric['count'])
    return metrics


def cpu_limit(job_type, spec):
    if job_type.endswith("map"):
        return spec.get("mapper", {}).get("cpu_limit", 1)
    elif job_type.endswith("reduce"):
        return spec.get("reducer", {}).get("cpu_limit", 1)
    return 1


def get_metric_dict_from_progress(progress, metric_path):
    result = progress
    for token in metric_path:
        result = result.get(token, {})
    return result.get("$$", {})


def iter_operation_metrics(operation_info):
    metrics = defaultdict(Metric)
    duration = datetime.strptime(operation_info["finish_time"], TIME_FORMAT) - datetime.strptime(operation_info["start_time"], TIME_FORMAT)
    yield "duration", Metric(duration.total_seconds(), duration.total_seconds(), duration.total_seconds(), 1)

    progress = operation_info.get("progress", "{}")
    if progress is None:
        logger.warning("Operation %s has no progress", operation_info["operation_id"])
        raise StopIteration

    progress = json.loads(operation_info.get("progress", "{}"))
    spec = json.loads(operation_info["spec"])

    if progress:
        if "job_statistics" in progress:
            job_statistics = progress["job_statistics"]

            basic_metrics = [
                ("user_job", "max_memory"),
                ("user_job", "current_memory", "rss"),
            ]
            for metric_path in basic_metrics:
                metric_dict = get_metric_dict_from_progress(job_statistics, metric_path)
                for job_type, metric in get_jobs_metric(metric_dict).items():
                    metrics[".".join(metric_path)] += metric

            time_total_metric_path = ("time", "total")
            time_total_metric_dict = get_metric_dict_from_progress(job_statistics, time_total_metric_path)
            for job_type, metric in get_jobs_metric(time_total_metric_dict).items():
                metrics[".".join(time_total_metric_path)] += metric
                metrics["cpu_quota_usage"] += metric * cpu_limit(job_type, spec)

        for sensor, metric in metrics.items():
            yield sensor, metric


def iter_metrics(operations):
    yield {"sensor": "operations"}, len(operations)

    metrics = defaultdict(Metric)
    for op in operations:
        for sensor, metric in iter_operation_metrics(op):
            metrics[sensor] += metric

    for sensor, metric in metrics.items():
        labels = {
            "sensor": sensor,
        }
        yield labels, metric.sum
