import logging
from datetime import datetime, timedelta
import pytz


logger = logging.getLogger(__name__)


DATE_FORMAT = "%Y-%m-%d"
TIME_FORMAT = "%Y-%m-%dT%H:%M:%S"


def get_list_of_daily_tables(from_dt, to_dt, daily_log_table_names, date_format=DATE_FORMAT):
    """
    >>> from_dt = datetime(year=2019, month=8, day=20)
    >>> to_dt = datetime(year=2019, month=8, day=21)
    >>> get_list_of_daily_tables(from_dt, to_dt, ['2019-08-19', '2019-08-20', '2019-08-21'])
    ['2019-08-20', '2019-08-21']
    >>> from_dt = datetime(year=2019, month=8, day=20)
    >>> to_dt = datetime(year=2019, month=8, day=22)
    >>> get_list_of_daily_tables(from_dt, to_dt, ['2019-08-19', '2019-08-20', '2019-08-21'])
    ['2019-08-20', '2019-08-21']
    >>> from_dt = datetime(year=2019, month=8, day=21)
    >>> to_dt = datetime(year=2019, month=8, day=22)
    >>> get_list_of_daily_tables(from_dt, to_dt, ['2019-08-19', '2019-08-20'])
    []
    >>> from_dt = datetime(year=2019, month=8, day=20, hour=8)
    >>> to_dt = datetime(year=2019, month=8, day=20, hour=9)
    >>> get_list_of_daily_tables(from_dt, to_dt, ['2019-08-19', '2019-08-20'])
    ['2019-08-20']
    """
    daily_table_names = []
    from_localized = pytz.utc.localize(from_dt)
    to_localized = pytz.utc.localize(to_dt)
    for table_name in daily_log_table_names:
        current_localized = pytz.timezone("Europe/Moscow").localize(datetime.strptime(table_name, date_format)).astimezone(pytz.utc)

        logger.debug("Check intersection of [%s, %s] and [%s, %s]", from_localized, to_localized, current_localized, current_localized + timedelta(days=1))
        if any([
            from_localized <= current_localized <= to_localized,
            from_localized <= current_localized + timedelta(days=1) <= to_localized,
            current_localized <= from_localized <= current_localized + timedelta(days=1),
            current_localized <= to_localized <= current_localized + timedelta(days=1),
        ]):
            daily_table_names.append(table_name)

    return daily_table_names


def get_list_of_tables(from_dt, to_dt, yt_client):
    daily_log_table_names = sorted(yt_client.list("//logs/yt-scheduler-log/1d"))

    logger.debug("Got daily tables: %s", daily_log_table_names)

    daily_log_tables = [
        "//logs/yt-scheduler-log/1d/{}".format(table_name)
        for table_name in get_list_of_daily_tables(from_dt, to_dt, daily_log_table_names)
    ]
    return daily_log_tables
