import codecs
import logging
import textwrap

from datetime import datetime, timedelta
from os import path

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import solomon
from sandbox.sandboxsdk import environments


LOG_PATH = 'logs/bs-proto-request-log/1d'
JSONP_TABLE_PATH = '//home/bs/Monitoring/Jsonp'
INVALID_JSON_TABLE_PATH = '//home/bs/Monitoring/Jsonp'

EXPORT_CLUSTER = 'hahn'

YQL_REQUEST = '''
    $is_jsonp = Re2::Match(@@Ya\\[\\d+\\]\\(.+@@);

    $skip_response = ($response) -> {{
        $body = String::RemoveFirst(String::RemoveFirst($response, "\\r"), "\\n");
        return $response == "" or String::StartsWith($body, '<?xml') or String::StartsWithIgnoreCase($body, '<VAST');
    }};

    $requests = (
        select
            *
        from
            `{log_path}`
        where
            Handler == 'meta' and
            ResponseBody is not null and
            ResponseCode == 200 and
            not $skip_response(ResponseBody)
    );

    select
        $is_jsonp(ResponseBody) as jsonp,
        (Yson::ParseJson(ResponseBody, Yson::Options(false as Strict, true as AutoConvert)) is null) as invalid_json,
        RequestID as requestid,
        ResponseBody as body
    from
        $requests
    where
        $is_jsonp(ResponseBody) or
        Yson::ParseJson(ResponseBody, Yson::Options(false as Strict, true as AutoConvert)) is null;
'''


def handle(err):
    return (u''.join(['\\' + hex(ord(err.object[i]))[1:] for i in range(err.start, err.end)]), err.end)
codecs.register_error('backslashreplace', handle)


def parse_type(s):
    if s == 'false':
        return False
    elif s == 'true':
        return True
    return s


class JsonpTrafficMonitor(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql', version='1.2.91'),
            environments.PipEnvironment('solomon'),
        )
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        jsonp_value = 0
        invalid_json_value = 0

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('YQL parameters') as yql_params:
            yql_token = sdk2.parameters.YavSecret('YQL token secret', default='sec-01dh95zr6cgqtsekwka08apdax', required=True)
            yql_cluster = sdk2.parameters.String('YQL cluster', default='hahn', required=True)
            write_yql = sdk2.parameters.Bool("Select invalid JSON / JSONP", default=True, required=True)
            with write_yql.value[True]:
                jsonp_table = sdk2.parameters.String('YT path to upload JSONP responses', default=JSONP_TABLE_PATH, required=True)
                invalid_json_table = sdk2.parameters.String('YT path to upload invalid JSON responses', default=INVALID_JSON_TABLE_PATH, required=True)

        with sdk2.parameters.Group('Solomon parameters') as solomon_params:
            write_solomon = sdk2.parameters.Bool("Export data to Solomon", default=True, required=True)
            with write_solomon.value[True]:
                solomon_token = sdk2.parameters.YavSecret('Solomon token secret', default='sec-01dh95zr6cgqtsekwka08apdax', required=True)
                solomon_project = sdk2.parameters.String('Solomon project', default='yabs', required=True)
                solomon_cluster = sdk2.parameters.String('Solomon cluster', default='yabs', required=True)
                solomon_service = sdk2.parameters.String('Solomon service', default='jsonp_traffic_monitor', required=True)

    def yt_connect(self, yt_cluster):
        import yt.wrapper as yt
        try:
            cfg = {
                'tabular_data_format': yt.JsonFormat(control_attributes_mode='row_fields'),
                'detached': False,
                'token': self.Parameters.yql_token.data()['yql_token'],
                'proxy': {'url': yt_cluster},
            }
            client = yt.YtClient(config=cfg)
            logging.info('Successfully connected to {}'.format(yt_cluster))
            return client
        except Exception as e:
            logging.info('Failed to connect to {}: {}'.format(yt_cluster, e))

    def _create_client(self):
        from yql.api.v1.client import YqlClient
        return YqlClient(db=self.Parameters.yql_cluster, token=self.Parameters.yql_token.data()['yql_token'])

    def _select_date(self):
        process_date = datetime.utcnow() - timedelta(days=1)
        process_date = process_date.replace(hour=0, minute=0, second=0)
        process_date = process_date.strftime('%Y-%m-%d')
        self.Context.process_date = process_date

    def _form_query(self):
        query = YQL_REQUEST.format(log_path=path.join(LOG_PATH, self.Context.process_date))
        return textwrap.dedent(query)

    def _run_yql_task(self):
        def callback(request):
            self.Context.operation_id = request.operation_id
        request = self.yql_client.query(self._form_query(), syntax_version=1)
        request.run(pre_start_callback=callback)

    def _wait_yql(self):
        from yql.client.operation import YqlOperationStatusRequest

        operation_id = self.Context.operation_id
        status = YqlOperationStatusRequest(operation_id)
        status.run()
        if status.status in status.IN_PROGRESS_STATUSES:
            raise sdk2.WaitTime(60)
        if status.status != 'COMPLETED':
            raise TaskFailure('YQL query failed')

    def _get_results(self):
        from yql.client.operation import YqlOperationResultsRequest

        operation_id = self.Context.operation_id
        results = YqlOperationResultsRequest(operation_id)
        results.run()

        table = next(iter(results.get_results()))
        columns = []
        for column_name, column_type in table.columns:
            columns.append(column_name)

        table.fetch_full_data()
        data = []
        for row in table.rows:
            data.append(dict(zip(columns, map(parse_type, row))))
        return data

    def _export_to_solomon(self):
        if not self.Parameters.write_solomon:
            return

        token = self.Parameters.solomon_token.data()['solomon_token']

        common_labels = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }

        labels = ['jsonp', 'invalid_json']
        values = [self.Context.jsonp_value, self.Context.invalid_json_value]
        process_date = datetime.strptime(self.Context.process_date, '%Y-%m-%d')
        sensors = [
            {
                'labels': {'sensor': label},
                'ts': process_date.strftime('%Y-%m-%dT%H:%M:%SZ'),
                'value': value,
            } for label, value in zip(labels, values)
        ]

        solomon.push_to_solomon_v2(token, common_labels, sensors)
        logging.info('Data exported to Solomon')

    def _export_to_yt(self, jsonp_rows, invalid_json_rows):
        if not self.Parameters.write_yql:
            return

        import yt.wrapper as yt

        def append_name(basepath, name):
            return path.join(basepath, '{}_{}'.format(name, self.Context.process_date))

        paths = [
            append_name(self.Parameters.jsonp_table, 'jsonp'),
            append_name(self.Parameters.invalid_json_table, 'invalid-json')
        ]
        data = [
            jsonp_rows,
            invalid_json_rows
        ]

        ytc = self.yt_connect(EXPORT_CLUSTER)
        for table, rows in zip(paths, data):
            if ytc.exists(table):
                logging.info('Table {} already exists, skipping'.format(table))
                continue

            ytc.create_table(
                table,
                ignore_existing=False,
                attributes={
                    'schema': [
                        {'name': 'requestid', 'type': 'string'},
                        {'name': 'body', 'type': 'string'}
                    ]
                })
            ytc.write_table(table, rows, format=yt.JsonFormat(attributes={'encode_utf8': False}))
            logging.info('Data wrote to {} ({} lines)'.format(table, len(rows)))

    def _push_results(self):
        jsonp_rows = []
        invalid_json_rows = []

        for row in self._get_results():
            result = {'requestid': row['requestid'], 'body': row['body'].decode('utf8', 'backslashreplace')}
            if row['jsonp']:
                jsonp_rows.append(result)
            elif row['invalid_json']:
                invalid_json_rows.append(result)

        self.Context.jsonp_value = len(jsonp_rows)
        self.Context.invalid_json_value = len(invalid_json_rows)

        self._export_to_solomon()
        self._export_to_yt(jsonp_rows, invalid_json_rows)

    def on_execute(self):
        self.yql_client = self._create_client()

        with self.memoize_stage.select_date:
            self._select_date()

        with self.memoize_stage.run_yql_task(commit_on_entrance=False):
            self._run_yql_task()

        self._wait_yql()

        with self.memoize_stage.push_results:
            self._push_results()

    @sdk2.header()
    def header(self):
        if self.status == ctt.Status.SUCCESS:
            return textwrap.dedent('''
                <table>
                    <tr>
                        <td>JSONP</td>
                        <td>{}</td>
                    </tr>
                    <tr>
                        <td>Invalid JSON</td>
                        <td>{}</td>
                    </tr>
                </table>'''.format(self.Context.jsonp_value,
                                   self.Context.invalid_json_value))
        else:
            return 'Task is not finished or failed'
