import json
from sandbox import sdk2
from sandbox.projects.yabs.monitoring.tasks.py3.PartnerShareMonitoring.lib.yql_helper import YQLHelper
from sandbox.projects.yabs.monitoring.tasks.py3.PartnerShareMonitoring.lib.startrek_helper import StartrekHelper
from sandbox.common.types.task import Status
from sandbox.common.types.notification import Transport, JugglerStatus


class BadRowsNotEmpty(Exception):
    pass


def flatten(lst):
    if type(lst) == list:
        ans = []
        for sublist in lst:
            ans += flatten(sublist)
        return ans
    else:
        return [lst]


class PartnerShareMonitoring(sdk2.Task):

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Tokens", collapse=True) as tokens:
            tokens = sdk2.parameters.YavSecret(
                "YAV tokens",
                default="sec-01f88883549gvnv08qatyzmwcd"
            )

        with sdk2.parameters.Group("Select parameters", collapse=False) as select_params:
            yt_cluster = sdk2.parameters.String(
                "YT cluster to get stats from",
                default="hahn"
            )
            rows_limit = sdk2.parameters.Integer(
                "Limit to the number of rows returned",
                default=1000
            )
            page_blacklist = sdk2.parameters.List(
                "Exclude given pages from monitoring",
                default=["1422884", "1422885"]
            )

        with sdk2.parameters.Group("YT tables location", collapse=False) as yt_tables_location:
            page_dsp_location = sdk2.parameters.String(
                "Location of PageDSP table in YT",
                default="//home/yabs/dict/PageDSP"
            )
            partner_page_location = sdk2.parameters.String(
                "Location of PartnerPage table in YT",
                default="//home/yabs/dict/PartnerPage"
            )
            page_location = sdk2.parameters.String(
                "Location of Page table in YT",
                default="//home/yabs/dict/Page"
            )
            tmp_folder = sdk2.parameters.String(
                "Location for tmp yql results",
                default="//home/yabs/tac-manager/yql_tmp"
            )

        with sdk2.parameters.Group("Startrek settings", collapse=False) as st_settings:
            st_queue = sdk2.parameters.String(
                "Startrek queue to create tickets",
                default="BSAUDIT",
                required=True
            )
            st_followers = sdk2.parameters.List(
                "Startrek ticket followers",
                default=[]
            )
            st_assignee = sdk2.parameters.String(
                "Startrek ticket assignee",
                default=None
            )
            st_components = sdk2.parameters.List(
                "Startrek ticket components",
                default="audit-tac"
            )

    def on_save(self):
        self.Parameters.notifications = [
            sdk2.Notification(
                [Status.FAILURE, Status.Group.BREAK],
                ["host=yabs_sandbox&service=PARTNER_SHARE_MONITORING"],
                Transport.JUGGLER,
                check_status=JugglerStatus.CRIT
            ),
            sdk2.Notification(
                [Status.SUCCESS],
                ["host=yabs_sandbox&service=PARTNER_SHARE_MONITORING"],
                Transport.JUGGLER,
                check_status=JugglerStatus.OK
            )
        ]

    def on_execute(self):
        tokens = self.Parameters.tokens.data()
        self.set_info("Getting YQL token from YAV")
        yql_token = tokens['yql_token']
        self.set_info("Gettng YQL Helper")
        yql_helper = YQLHelper(yql_token)
        self.set_info("Getting bad pages from YQL")
        bad_pages = yql_helper.get_bad_ad_inside_pages(
            yt_cluster=self.Parameters.yt_cluster,
            page_dsp_location=self.Parameters.page_dsp_location,
            partner_page_location=self.Parameters.partner_page_location,
            page_location=self.Parameters.page_location,
            tmp_folder=self.Parameters.tmp_folder,
            rows_limit=self.Parameters.rows_limit,
            page_blacklist=self.Parameters.page_blacklist
        )
        self.set_info(f"YQL query: {bad_pages.operation_id}")
        json_value = bad_pages.json["data"][0]["Write"][0]["Data"]
        self.set_info(json.dumps(json_value, indent=4))
        if json_value:
            st_helper = StartrekHelper(
                useragent="TAC Monitoring sandbox task",
                st_token=tokens["st_token"]
            )
            description = []
            description.append(f"Sandbox task https://sandbox.yandex-team.ru/task/{self.id}/view")
            description_text = "\n".join(str(flatten(value)) for value in json_value)
            description.append(
                f"%%\n{description_text}\n%%"
            )
            st_issue = st_helper.create_issue(
                queue=self.Parameters.st_queue,
                summary=f"TAC Monitoring failed in task {self.id}",
                description="\n".join(description),
                assignee=self.Parameters.st_assignee,
                followers=self.Parameters.st_followers,
                components=self.Parameters.st_components
            )
            if st_issue is None:
                self.set_info("Failed to create a Startrek issue")
            else:
                self.set_info(f"Created Startrek ticket {st_issue.key}")
            self.set_info("Result is not empty, task has failed")
            raise BadRowsNotEmpty()
        self.set_info("Result is empty, everything is OK")
