from sandbox.common.fs import get_unique_file_name
from sandbox.projects.yabs.qa.sut.utils import reserve_port

from sandbox import sdk2

import logging
import subprocess
import os
import imp
import json
import shutil
import gzip
import urllib2
import time
from base64 import b64encode

from concurrent import futures

from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated

IPV6_LOOPBACK_ADDR = '::1'


class NanpuSUT(ModuleBase):

    EXT_TAGS_TO_CONFIG_VARS_DICT = {
        'rtb': ['rtb_get_port', 'yabs_get_port'],
        'lbs': ['lbs_get_port'],
        'laas': ['laas_get_port'],
        'adfox': ['adfox_get_port']
    }

    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._active = False
        nanpu_package_path = get_unique_file_name(adapter.get_work_dir(), 'nanpu_package')
        package_postfix = os.path.basename(nanpu_package_path)
        os.makedirs(nanpu_package_path)
        with sdk2.helpers.ProcessLog(adapter._task_instance, logging.getLogger('bundle_unpack')) as process_log:
            subprocess.Popen(
                [
                    'tar',
                    '-xvf',
                    adapter.get_nanpu_resource_path(),
                    '-C',
                    nanpu_package_path
                ],
                stdout=process_log.stdout,
                stderr=subprocess.STDOUT
            ).wait()
        nanpu_package_path = os.path.join(nanpu_package_path, 'yabs-nanpu-package')
        archives_path = os.path.join(nanpu_package_path, 'linux')
        self._bundle_path = os.path.join(nanpu_package_path, 'bundle')
        os.makedirs(self._bundle_path)
        with sdk2.helpers.ProcessLog(adapter._task_instance, logging.getLogger('binary_unpack')) as process_log:
            subprocess.Popen(
                [
                    'tar',
                    '-xvf',
                    os.path.join(archives_path, 'nanpu.tgz'),
                    '-C', self._bundle_path
                ],
                stdout=process_log.stdout,
                stderr=subprocess.STDOUT
            ).wait()
        with sdk2.helpers.ProcessLog(adapter._task_instance, logging.getLogger('static_unpack')) as process_log:
            subprocess.Popen(
                [
                    'tar',
                    '-xvf',
                    os.path.join(archives_path, 'static.tgz'),
                    '-C', self._bundle_path
                ],
                stdout=process_log.stdout,
                stderr=subprocess.STDOUT
            ).wait()
        find_module_tuple = imp.find_module('plugin', [nanpu_package_path, ])
        nanpu_plugin_module = imp.load_module('plugin', *find_module_tuple)
        self._nanpu_config = nanpu_plugin_module.NANPU_CONFIG
        self._resolve_dict = nanpu_plugin_module.RESOLVE
        logs_dir = os.path.join(adapter.get_logs_dir(), package_postfix)
        os.makedirs(logs_dir)
        for key in self._resolve_dict:
            split_name_list = key.rsplit('_', 1)
            if len(split_name_list) == 2 and split_name_list[1] == 'log':
                self._resolve_dict[key] = os.path.join(logs_dir, '{log_name}.log'.format(log_name=split_name_list[0]))
        for hostname in nanpu_plugin_module.HOSTNAMES:
            self._resolve_dict[hostname] = '{{ {loopback} }}'.format(loopback=IPV6_LOOPBACK_ADDR)
        base_dir = adapter.get_base_dir()
        self._resolve_dict.update({
            'resource_files_mraid': os.path.join(base_dir, 'mraid.json'),
            'resource_files_feedback_map': os.path.join(base_dir, 'feedback.json'),
            'resource_files_feedback_social_map': os.path.join(base_dir, 'feedback_social.json'),
            'resource_files_nmb_banner_id_map': os.path.join(base_dir, 'dooh_creatives.json'),
            'resource_files_dooh_page_imp': os.path.join(base_dir, 'dooh_page_imp.json'),
            'resource_files_dooh_operators': os.path.join(base_dir, 'dooh_operators.json'),
            'resource_files_debug_tokens': os.path.join(base_dir, 'debug_tokens.json'),
            'resource_files_dooh_display_d': os.path.join(base_dir, 'dooh_display_d.json'),
            'resource_files_indoor_page_imp': os.path.join(base_dir, 'indoor_page_imp.json'),
            'resource_files_network_whitelist': os.path.join(base_dir, 'network_whitelist.json'),
        })
        static_dir = os.path.join(self._bundle_path, 'static')
        self._resolve_dict.update({
            'mime_types_filename': os.path.join(static_dir, 'mime.types'),
            'demo_path': static_dir,
            'resource_path': os.path.join(static_dir, 'img'),
            'xml_path': os.path.join(static_dir, 'crossdomain'),
            'mappings_path': os.path.join(static_dir, 'mappings')
        })
        self._resolve_dict.update({
            'log_external_requests': 'true',
            'rtb_links_count': 950,
            'contour_no': 0,
            'DOOH_FACE_PP_KEY': 'fake_1',
            'DOOH_FACE_PP_SECRET': 'fake_2',
            'DOOH_FIND_FACE_TOKEN': 'fake_3',
            'DOOH_ADFOX_S2S_KEY': 'fake_4',
            'NANPU_TVM_SECRET': 'dummy',
            'REWARD_BROKER_NOTIFY_KEY': b64encode('dummy'),
            'tvm_bigb_id': 2001337,
            'nanpu_tvm_self_id': 2010526,
            'tvm_navistopwatch_id': 5001,
            'is_dooh_contour': 'false',
        })
        services_ports=self.adapter.get_cache_daemon_ports_map()
        self._cachedaemon = CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=self.adapter.get_cache_daemon_executable_path(),
            dump_path=self.adapter.get_cache_daemon_stub_path(),
            data_dir=get_unique_file_name(self.adapter.get_work_dir(), 'cache_daemon_nanpu_data'),
            log_subdir=get_unique_file_name(self.adapter.get_logs_dir(), 'cache_daemon_nanpu_logs'),
            start_on_creation=False,
            key_header=self.adapter.get_cache_daemon_key_headers(),
            services=None,
            services_ports=services_ports,
        )
        cachedaemon_ports = self._cachedaemon.get_ports_by_tag()
        self._resolve_dict.update({
            config_var: cachedaemon_ports.get(tag, services_ports[tag]) for tag, config_list in self.EXT_TAGS_TO_CONFIG_VARS_DICT.iteritems() for config_var in config_list
        })
        self.provide_bases(base_dir)

    def provide_bases(self, base_dir, max_workers=4):
        base_iterator = self.adapter.get_base_iterator()
        with futures.ThreadPoolExecutor(max_workers=max_workers) as pool:
            future_list = []
            while not base_iterator.is_exhausted():
                future_list.append(pool.submit(self.provide_one_base, base_dir, base_iterator))
            for future in futures.as_completed(future_list):
                result = future.result()
                if result is not None:
                    logging.info('Base {base} complete'.format(base=result))

    def wait_ping(self, timeout=300):
        if not self._active:
            raise RuntimeError('wait_ping was used to check on inactive service')
        port = self.get_port()
        deadline = time.time() + timeout
        pause = 0.25
        request = urllib2.Request("http://localhost:{}/status".format(port))
        logging.info('Port: {}'.format(port))
        while True:
            error_msg = ''
            try:
                urllib2.urlopen(request, timeout=10)
            except urllib2.HTTPError as e:
                error_msg = "/status returned %d" % e.code
            except Exception as e:
                error_msg = "/status exception: %s" % (str(e))
            else:
                logging.info("responded, ok")
                break

            now = time.time()
            if now > deadline:
                raise RuntimeError(
                    "No 200 response after %s seconds (port %s): %s" % (timeout, port, error_msg)
                )
            logging.info(error_msg)
            time.sleep(pause)
            pause *= 1.5
            logging.info("Retrying /status ...")

    @staticmethod
    def provide_one_base(base_dir, base_iterator):
        try:
            base_path = base_iterator.next()
        except StopIteration:
            return None
        base_name = os.path.basename(base_path)
        base_name_unpacked = base_name[:-3]
        with gzip.open(base_path, 'rb') as f_in:
            with open(os.path.join(base_dir, base_name_unpacked), 'wb') as f_out:
                shutil.copyfileobj(f_in, f_out)
        return base_name_unpacked

    def __enter__(self):
        self._sockets = []
        for port_name in ('nanpu_port', 'status_port', 'monitor_port'):
            if port_name == 'nanpu_port' and self.adapter.get_nanpu_port() is not None and self.adapter.get_nanpu_port() != 0:
                port = self.adapter.get_nanpu_port()
                self._resolve_dict[port_name] = port
            else:
                port, socket = reserve_port()
                self._sockets.append(socket)
                self._resolve_dict[port_name] = port
        config_path = os.path.join(self._bundle_path, 'config')
        with open(config_path, 'w') as config_file:
            config_file.write(self._nanpu_config.format(**self._resolve_dict))
        command_line = [
            os.path.join(self._bundle_path, 'nanpu'),
            'run',
            config_path
        ]
        self._process_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'nanpu.run')
        self._process_log_context.__enter__()
        self._cachedaemon.__enter__()
        self._process = subprocess.Popen(command_line,
                                         stdout=self._process_log_context.stdout,
                                         stderr=self._process_log_context.stderr)
        logging.info(json.dumps(self._resolve_dict, indent=4))
        self._active = True
        self.wait_ping()
        # logging.info('Sleeping')
        # time.sleep(300)
        return self

    def __exit__(self, *args):
        self._process.kill()
        self._process.communicate()
        self._process_log_context.__exit__(*args)
        self._process_log_context = None
        self._cachedaemon.__exit__(*args)
        self._active = False

    def get_port(self):
        if self._active:
            return self._resolve_dict['nanpu_port']
        else:
            return None

    def __del__(self):
        pass
