import ast
import json
import logging
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.nanpu import BS_NANPU_BIN_BASE_GEN, BS_NANPU_BASE_GEN_YT_PATH
from sandbox.projects.yabs.nanpu.tasks.ProcessBase import BsNanpuSingleBaseProcess
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp


class BsNanpuBaseGen(sdk2.Task):
    """Base generation task for Nanpu"""
    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('yandex-yt'),)

    class Parameters(sdk2.Task.Parameters):
        description = 'Generation of Nanpu bases.'

        source_file = sdk2.parameters.Resource('BaseGen binary', resource_type=BS_NANPU_BIN_BASE_GEN, required=True)
        yt_path = sdk2.parameters.Resource('YT path', resource_type=BS_NANPU_BASE_GEN_YT_PATH, required=True)
        yt_token = sdk2.parameters.String('yt-token name', default='nanpu_tests_yt_token')
        arcadia_patch = sdk2.parameters.String('Apply patch', default='')

        with sdk2.parameters.Output:
            base_resources = sdk2.parameters.Resource('Base resources', multiple=True)

    class Context(sdk2.Task.Context):
        children_enqueued = False
        yt_json = dict()
        base_resources = []

    def on_execute(self):
        import yt.wrapper as yt
        yt_path = sdk2.ResourceData(self.Parameters.yt_path).path.open().read().strip()
        yt.config['token'] = try_get_from_vault(self, self.Parameters.yt_token)
        yt.config['proxy']['url'] = 'hahn'
        logging.info('YT-path: ' + yt_path)

        def acquire_lock(self, yt_path):
            with yt.Transaction():
                if not yt.exists(yt_path):
                    yt.create('file', yt_path, recursive=True, attributes={'locked': -1})
                yt.lock(yt_path, waitable=True)

                locked_id = yt.get(yt_path + '/@locked')
                if locked_id == self.id:
                    logging.info('Lock is already owned.')
                elif locked_id != -1 and sdk2.Task[locked_id].status not in ctt.Status.Group.FINISH:
                    logging.info('Lock is busy. Waiting for owner of lock.')
                    raise sdk2.WaitTask(sdk2.Task[locked_id], ctt.Status.Group.FINISH, timeout=1800)
                else:
                    logging.info('Acquiring lock.')
                    yt.set(yt_path + '/@locked', self.id)

        def release_lock(self, yt_path):
            with yt.Transaction():
                yt.lock(yt_path, waitable=True)
                if yt.get(yt_path + '/@locked') == self.id:
                    logging.info('Releasing lock')
                    yt.set(yt_path + '/@locked', -1)
                    logging.info('yt_path has been released.')

        try:
            if not self.Context.children_enqueued:
                acquire_lock(self, yt_path)

                binary_path = str(sdk2.ResourceData(self.Parameters.source_file).path)
                args = [binary_path, 'base_list']
                with sdk2.helpers.ProcessLog(self, logger="binary") as pl:
                    pl.logger.propagate = 1
                    subprocess = sp.Popen(args, stdout=sp.PIPE, stderr=pl.stderr)
                    subprocess.wait()
                    binary_base_list = ast.literal_eval(subprocess.stdout.read())
                logging.info('Bases from binary: %s' % str(binary_base_list))

                yt_raw = yt.read_file(yt_path).read()
                self.Context.yt_json = json.loads(yt_raw if len(yt_raw) else '{}')
                yt_base_list = self.Context.yt_json.keys()
                logging.info('Bases from YT: %s' % str(yt_base_list))

                bases_to_gen = list(set(binary_base_list) - set(yt_base_list))
                logging.info('Bases to gen: %s' % str(bases_to_gen))

                if len(bases_to_gen) == 0:
                    logging.info('All required bases already exist. Skipping generation.')
                    base_resources = [sdk2.Resource[id] for id in self.Context.yt_json.values()]
                    self.Context.base_resources.extend(resource.id for resource in base_resources)
                    self.Parameters.base_resources = base_resources
                    return

                child_tasks = []
                for name in bases_to_gen:
                    task = BsNanpuSingleBaseProcess(
                        self,
                        description='Base generation task for base "{}" runned by {}'.format(name, self.id),
                        notifications=self.Parameters.notifications,
                        create_sub_task=False,
                        source_file=self.Parameters.source_file,
                        basename=name,
                        yt_token=self.Parameters.yt_token
                    )

                    task.save().enqueue()
                    logging.info('Enqueued task for %s base.' % name)
                    child_tasks.append(task.id)

                logging.info('Waiting for children output.')
                self.Context.children_enqueued = True
                raise sdk2.WaitOutput({task: 'base_resource' for task in child_tasks}, wait_all=True)

            logging.info('Children finished.')

            base_resources = [sdk2.Resource[id] for id in self.Context.yt_json.values()]
            base_resources += [subtask.Parameters.base_resource for subtask in self.find()]
            generated_ids = {subtask.Parameters.basename: subtask.Parameters.base_resource.id for subtask in self.find()}
            logging.info('Generated bases: %s' % str(generated_ids))

            self.Context.yt_json.update(generated_ids)

            if len(self.Parameters.arcadia_patch) == 0:
                logging.info('Updating yt data.')
                yt.write_file(yt_path, json.dumps(self.Context.yt_json))

            self.Context.base_resources.extend(resource.id for resource in base_resources)
            self.Parameters.base_resources = base_resources
        except sdk2.Wait as e:
            raise e
        except:
            logging.info('Caught exception. Trying to release yt_path.')
            release_lock(self, yt_path)
            raise
        else:
            logging.info('Finishing task.')
            release_lock(self, yt_path)
