import logging
import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.common.errors import ResourceNotFound
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.nanpu import BS_NANPU_BASE_GEN_YT_PATH
from sandbox.sandboxsdk import environments


class BsNanpuManageYtPath(sdk2.Task):
    """Task generating fresh YT path for Nanpu base dumps and deleting old ones."""
    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('yandex-yt'),)

    class Parameters(sdk2.Task.Parameters):
        description = 'Generates path for ids of Nanpu base dumps in YT.'
        dirpath = sdk2.parameters.String('Dirpath to clean', default='//home/yabs-nanpu-sandbox-tests/bases')
        file_template = sdk2.parameters.String('Template being completed with the task id', default='//home/yabs-nanpu-sandbox-tests/bases/bases_{}.json')
        yt_token = sdk2.parameters.String('yt-token name', default='nanpu_tests_yt_token')
        testenv_switch_trigger = sdk2.parameters.Bool('Switch Testenv to generated resources', default=False)

        with sdk2.parameters.Output:
            yt_path = sdk2.parameters.Resource('YT path', resource_type=BS_NANPU_BASE_GEN_YT_PATH)

    def on_execute(self):
        import yt.wrapper as yt
        yt.config['token'] = try_get_from_vault(self, self.Parameters.yt_token)
        yt.config['proxy']['url'] = 'hahn'

        yt_path = self.Parameters.file_template.format(self.id).strip()
        logging.info('YT path to create is %s' % yt_path)
        with open('yt_path.txt', 'w') as file:
            file.write(yt_path)

        attrs = {'ttl': 14}
        if self.Parameters.testenv_switch_trigger:
            attrs.update({'testenv_switch_trigger_nanpu': self.id})
        self.Parameters.yt_path = BS_NANPU_BASE_GEN_YT_PATH(self, 'YT path', 'yt_path.txt', **attrs)
        sdk2.ResourceData(self.Parameters.yt_path).ready()
        logging.info('Created resource with YT path.')

        with yt.Transaction():
            if not yt.exists(yt_path):
                yt.create('file', yt_path, recursive=True, attributes={'locked': -1})
            yt.set(yt_path + '/@sb_resource_id', self.Parameters.yt_path.id)
        logging.info('Initiated path in YT.')

        yt_objects = yt.list(self.Parameters.dirpath, absolute=True, attributes=['sb_resource_id'])
        logging.info('Got list of child nodes of length %d.' % len(yt_objects))
        for obj in yt_objects:
            if len(obj.attributes):
                logging.info('%s is a json with bases.' % str(obj))
                sb_resource_id = obj.attributes['sb_resource_id']
                try:
                    if sdk2.Resource[sb_resource_id].state in [ctr.State.BROKEN, ctr.State.DELETED]:
                        logging.info('Resource %d corresponding to this path is DELETED or BROKEN. Deleting path.' % sb_resource_id)
                        yt.remove(str(obj))
                except ResourceNotFound:
                    logging.info('Resource %d corresponding to this path was not found. Deleting path.' % sb_resource_id)
                    yt.remove(str(obj))
        logging.info('Finished succesfully.')
