import logging
import datetime
from sandbox.common import errors
from sandbox import sdk2


class PantherImportStatisticalIndex(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 12 * 60 * 60

        with sdk2.parameters.Group("YT params", collapse=False) as yt_settings:
            yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)

        with sdk2.parameters.Group("YQL params", collapse=False) as yql_settings:
            yql_token_vault_name = sdk2.parameters.String("YQL Token vault name", default="yql_token", required=True)

        with sdk2.parameters.Group("Index params", collapse=False) as index_settings:
            index_result_table_path = sdk2.parameters.String("Path to result yt table", required=True)
            index_sampling = sdk2.parameters.Bool("Use sampling", default=False, required=True)

            with index_sampling.value[True]:
                index_sampling_percent = sdk2.parameters.Float("Sampling percent", default=0.01, required=True)

            with sdk2.parameters.RadioGroup("Statistics period type") as mode:
                mode.values["last_n"] = mode.Value("Last N hours", default=True)
                mode.values["manual"] = mode.Value("Manual mode")

            with mode.value["last_n"]:
                index_time_offset = sdk2.parameters.Integer('Time offset', default=12, required=True)
                index_interval_hours = sdk2.parameters.Integer('Statistics period in hours', default=24, required=True)

            with mode.value["manual"]:
                start_date = sdk2.parameters.StrictString(
                    "Start date YYYY-MM-DDTHH-MM-SS",
                    regexp=r"\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}",
                    required=True)
                end_date = sdk2.parameters.StrictString(
                    "End date YYYY-MM-DDTHH-MM-SS",
                    regexp=r"\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}",
                    required=True)

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from library.python import resource

        logging.info("Started")
        self.yql_token = sdk2.task.Vault.data(self.author, self.Parameters.yql_token_vault_name)
        self.query_template = resource.find('sandbox/projects/yabs/panther/run_yql_builder/build_index.yql')

        start_sql = ''
        end_sql = ''
        if self.Parameters.mode == "last_n":
            end = datetime.datetime.now() - datetime.timedelta(hours=self.Parameters.index_time_offset)
            start = end - datetime.timedelta(hours=self.Parameters.index_interval_hours)
            start_sql = start.strftime('%Y-%m-%dT%H:%M:%S')
            end_sql = end.strftime('%Y-%m-%dT%H:%M:%S')
        else:
            start_sql = self.Parameters.start_date
            end_sql = self.Parameters.end_date

        logging.info('Start date {}, end date {}'.format(start_sql, end_sql))
        if end_sql < start_sql:
            raise errors.TaskError('end date < start date')

        query = self.query_template.format(
            cluster=self.Parameters.yt_cluster,
            result_table=self.Parameters.index_result_table_path,
            start_date=start_sql,
            end_date=end_sql,
            sampling='TABLESAMPLE BERNOULLI({})'.format(self.Parameters.index_sampling_percent) if self.Parameters.index_sampling else ''
        )

        with YqlClient(token=self.yql_token) as yql_client:
            yql_query = yql_client.query(query=query, title="YQL PantherCS: import yql-based index")
            yql_query.run()
            yql_query.get_results(wait=True)
            if not yql_query.is_success:
                raise errors.TaskError('YQL query failed: {}'.format(yql_query.status))
