import logging
import os

from sandbox.common import errors
from sandbox import sdk2


class PantherShareBases(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        push_tasks_resource = True

        with sdk2.parameters.Group("YT params", collapse=False) as yt_params:
            yt_cluster = sdk2.parameters.String("YT cluster with tables", default="hahn", required=True)
            yt_token_vault_name = sdk2.parameters.String("YT Token vault name", default="yt_token", required=True)

        with sdk2.parameters.Group("Transport params", collapse=True) as transport_params:
            transport_yt_cluster = sdk2.parameters.String("Transport YT cluster", default="locke", required=True)
            transport_yt_path = sdk2.parameters.String("Path to yt transport dir", default="//tmp/transport_panther", required=True)
            transport_force_push = sdk2.parameters.Bool("Force push bases", default=False, required=False)
            transport_enable_fastbone = sdk2.parameters.Bool("Turn on fastbone option of yt sky_share", default=True, required=True)

        with sdk2.parameters.Group("Share params", collapse=True) as share_params:
            share_bases = sdk2.parameters.JSON('Bases to share {{"<base_name>": "<path_on_yt>"}}')
            share_mtime = sdk2.parameters.Integer("Mtime", default=None, required=True)

        with sdk2.parameters.Output():
            output_bases_info = sdk2.parameters.JSON('Transport info')

    def on_execute(self):
        logging.info('Share panther bases')
        logging.info(self.Parameters.push_tasks_resource)

        self._fill_bases_info_and_share_tables()
        self._fill_transport_info()
        self.Parameters.output_bases_info = self.Context.bases_info

    def _fill_bases_info_and_share_tables(self):
        from yt.wrapper import YtClient
        client = YtClient(
            proxy=self.Parameters.yt_cluster,
            token=sdk2.task.Vault.data(self.author, self.Parameters.yt_token_vault_name))
        self.Context.bases_info = {}

        share_bases = self.Parameters.share_bases
        for base in share_bases:
            self.Context.bases_info[base] = {
                "yt_path": share_bases[base],
                "mtime": self.Parameters.share_mtime,
                "torrent": self._share_table(client, share_bases[base]),
            }

    def _share_table(self, client, base_table_path):
        if not client.exists(base_table_path):
            raise errors.TaskError('Base blob table {} does not exist'.format(base_table_path))

        logging.info('Sharing {table} with enable_fastbone={fastbone}'.format(table=base_table_path, fastbone=self.Parameters.transport_enable_fastbone))
        return client.sky_share(base_table_path, enable_fastbone=self.Parameters.transport_enable_fastbone)

    def _fill_transport_info(self):
        from yt.wrapper import YtClient

        client = YtClient(
            proxy=self.Parameters.transport_yt_cluster,
            token=sdk2.task.Vault.data(self.author, self.Parameters.yt_token_vault_name))
        with client.Transaction():
            bases_info = self.Context.bases_info
            for base, base_info in bases_info.iteritems():
                logging.info('Fill transport info for base {}'.format(base))
                path = os.path.join(self.Parameters.transport_yt_path, base)
                path_info = os.path.join(path, '@_base_info')

                client.create(
                    'document', path, recursive=True, ignore_existing=True,
                    attributes={'_base_info': {}}
                )
                client.lock(path, waitable=True, wait_for=60)

                old_info = client.get(path_info, attributes=['_base_info'])
                logging.info('Base {} old_info: {}'.format(base, old_info))
                logging.info('Base {} base_info: {}'.format(base, base_info))

                if self.Parameters.transport_force_push or 'mtime' not in old_info or old_info['mtime'] < base_info['mtime']:
                    client.create(
                        'document', path, recursive=True, ignore_existing=False, force=True,
                        attributes={'_base_info': base_info}
                    )
                else:
                    logging.info('Skip base {} due to mtime check'.format(base))
