import logging
from sandbox.common import errors
from sandbox import sdk2


class PantherUpdateBannersState(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3 * 60 * 60

        with sdk2.parameters.Group("YT params", collapse=False) as yt_settings:
            yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)
            yt_token_vault_name = sdk2.parameters.String("YT Token vault name", default="yt_token", required=True)

        with sdk2.parameters.Group("YQL params", collapse=False) as yql_settings:
            yql_token_vault_name = sdk2.parameters.String("YQL Token vault name", default="yql_token", required=True)

        with sdk2.parameters.Group("State params", collapse=False) as index_settings:
            state_import_table_path = sdk2.parameters.String("Path to index import table", required=True)
            state_table_path = sdk2.parameters.String("Path to state table", required=True)
            state_banners_ttl = sdk2.parameters.Integer("Delete banners after N days without updates", default=2, required=True)

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from library.python import resource

        logging.info("Started")
        self.yql_token = sdk2.task.Vault.data(self.author, self.Parameters.yql_token_vault_name)
        self.query_template = resource.find('sandbox/projects/yabs/panther/update_banners_state/update_banners_state.yql')

        from yt.wrapper import YtClient

        client = YtClient(
            proxy=self.Parameters.yt_cluster,
            token=sdk2.task.Vault.data(self.author, self.Parameters.yt_token_vault_name))

        if not client.exists(self.Parameters.state_table_path):
            logging.info('Create output bases path: {}'.format(self.Parameters.state_table_path))
            client.create(
                'table',
                path=self.Parameters.state_table_path,
                recursive=True,
                ignore_existing=True,
                attributes={
                    'schema': [
                        {'name': 'BannerID', 'type': 'int64', 'required': True, 'sort_order': 'ascending'},
                        {'name': 'UpdateTime', 'type': 'date', 'required': True},
                    ],
                })

        query = self.query_template.format(
            cluster=self.Parameters.yt_cluster,
            state_table=self.Parameters.state_table_path,
            import_table=self.Parameters.state_import_table_path,
            banners_ttl=str(self.Parameters.state_banners_ttl),
        )

        with YqlClient(token=self.yql_token) as yql_client:
            yql_query = yql_client.query(query=query, title='YQL PantherCS: update banners state')
            yql_query.run()
            yql_query.get_results(wait=True)
            if not yql_query.is_success:
                raise errors.TaskError('YQL query failed: {}'.format(yql_query.status))
