from __future__ import unicode_literals
from __future__ import print_function

import os
import sys
import argparse
from sandbox.projects.yabs.partner_share.lib.config.config import (
    get_config,
    get_daemons,
)
from sandbox.projects.yabs.partner_share.lib.integratest.integratest_lib import (
    Integratest,
    full_flow,
)
from sandbox.projects.yabs.partner_share.lib.sandbox_helper import SandboxHelper
from sandbox.projects.yabs.partner_share.lib.st_helper import (
    get_queue_from_issue,
    get_sandbox_link,
)
from sandbox.projects.yabs.partner_share.lib.utils import (
    Color
)
import logging

logging.basicConfig(level=logging.WARN, format='%(asctime)s %(message)s')
l = logging.getLogger(__name__)
config = get_config()


class Unbuffered(object):
    def __init__(self, stream):
        self.stream = stream

    def write(self, data):
        self.stream.write(data)
        self.stream.flush()

    def writelines(self, datas):
        self.stream.writelines(datas)
        self.stream.flush()

    def __getattr__(self, attr):
        return getattr(self.stream, attr)


sys.stdout = Unbuffered(sys.stdout)


def get_st_token():
    token = ''
    token_path = os.path.expanduser("~/.startrek/token")
    if os.path.exists(token_path):
        with open(token_path, 'r') as fil:
            token = fil.read().replace('\n', '')
            if token:
                print('Using ' + token_path + ' file for Startrek token')
    else:
        raise RuntimeError(
            'Please put startrek token to ~/.startrek/token from '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=5f671d781aca402ab7460fde4050267b'
        )
    return token


def get_ok_token():
    token = ''
    token_path = os.path.expanduser("~/.ok/token")
    if os.path.exists(token_path):
        with open(token_path, 'r') as fil:
            token = fil.read().replace('\n', '')
            if token:
                print('Using ' + token_path + ' file for OK token')
    else:
        raise RuntimeError(
            'Please put OK token to ~/.ok/token from '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=0137c50bcf1e45afaa7a9fdd8d17fc72'
        )
    return token


def get_yql_token():
    token = os.environ.get('YQL_TOKEN')
    if token:
        print('Using YQL_TOKEN environment variable for YQL token')
    if not token:
        token = os.environ.get('YT_TOKEN')
        if token:
            print('Using YT_TOKEN environment variable for YQL token')
    if not token:
        token_path = os.path.expanduser("~/.yt/token")
        if os.path.exists(token_path):
            with open(token_path, 'r') as fil:
                token = fil.read().replace('\n', '')
                if token:
                    print('Using ' + token_path + ' file for YQL token')
    if not token:
        raise RuntimeError(
            'Cannot find YQL token in environment variables YT_TOKEN or YQL_TOKEN or in file ~/.yt/token '
            'Please get it from https://yql.yandex-team.ru'
        )
    return token


def get_sandbox_token():
    token = ''
    token_path = os.path.expanduser("~/.sandbox/token")
    if os.path.exists(token_path):
        with open(token_path, 'r') as fil:
            token = fil.read().replace('\n', '')
            if token:
                print('Using ' + token_path + ' file for sandbox token')
    else:
        raise RuntimeError(
            'Please put Sandbox token to ~/.sandbox/token from '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=968e0e6d85d647feb327d893a42cf26f&optional_scope=sandbox:api'
        )
    return token


def show_tasks(task_type, queue, sandbox_helper):
    print(task_type, queue, 'tasks:')
    tasks = sandbox_helper.get_tasks(task_type, queue, limit=100)['items']
    for task in tasks[:12]:
        color = Color.WHITE
        if task['status'] in ('SUCCESS'):
            color = Color.GREEN
        if task['status'] in ('ENQUEUED'):
            color = Color.WHITE
        if task['status'] == 'DRAFT':
            color = Color.GRAY
        if task['status'] in ('EXCEPTION', 'FAILURE', 'STOPPED'):
            color = Color.RED
        if task['status'] == 'EXECUTING':
            color = Color.BLUE
        print(
            get_sandbox_link(task['id']),
            task['time']['updated'],
            task['requirements']['tasks_resource'],
            color + task['status'] + Color.END
        )
    return tasks


def main():
    parser = argparse.ArgumentParser(description="TacMan integration testing")
    parser.add_argument(
        "--start-stage",
        help="Will start with this stage",
        choices=full_flow(config),
        default='FILTER'
    )
    parser.add_argument(
        "--end-stage",
        help="Will end with this stage",
        choices=full_flow(config),
        default='APPROVE_RESUME'
    )
    parser.add_argument(
        "--ticket",
        help="Choose ticket to run test",
        default='TESTTACCHANGES-74'
    )
    args = parser.parse_args()

    sandbox_helper = SandboxHelper(get_sandbox_token())

    for task_type in get_daemons(config):
        show_tasks(task_type, get_queue_from_issue(args.ticket), sandbox_helper)

    integratest = Integratest(
        config=config,
        start_stage=args.start_stage,
        end_stage=args.end_stage,
        ticket=args.ticket,
        st_token=get_st_token(),
        yql_token=get_yql_token(),
        ok_token=get_ok_token(),
        yt_cluster='hahn',
        sandbox_token=get_sandbox_token(),
    )
    integratest.run_integratest()

if __name__ == "__main__":
    main()
