from six import iteritems
from sandbox.common import fs

MAX_PARTNER_SHARE_FIELD = 'tacman_max_partner_share'
APPROVAL_TYPE_FIELD = 'tacman_approval_type'
APPROVAL_STAGES_FIELD = 'tacman_approve_stages'


def get_daemons(config):
    return [
        task_type
        for task_type in config['task_types']
        if config['task_types'][task_type].get('daemon')
    ]


def get_config():
    import json

    config = json.loads(fs.read_file('sandbox/projects/yabs/partner_share/lib/config/config.json'))
    validate_config(config)
    return config


def render_spec(
    config,
    stage_name,
    operation_name,
    task,

    yt_cluster,
    chyt_cluster,
    yql_token,
    yt_root,

    issue,
    direct_issue,
    filters,
    ignore_partner_share_above=1000000,
    dont_run_yql=False,
):
    from yt.wrapper import ypath_join
    import jinja2

    spec = config['operations'][operation_name]

    spec['task'] = task
    spec['yt_cluster'] = yt_cluster
    spec['chyt_cluster'] = chyt_cluster
    spec['yql_token'] = yql_token
    spec['yt_root'] = yt_root
    spec['issue'] = issue
    spec['direct_issue'] = direct_issue
    spec['ignore_partner_share_above'] = ignore_partner_share_above
    spec['filters'] = filters
    spec['dont_run_yql'] = dont_run_yql
    spec['constants'] = config['constants']

    stage = config['stages'][stage_name]

    issue_path = ypath_join(spec['constants']['TACMAN_REQUESTS_DIR'], spec['issue'])
    if 'stage_folder' in stage and stage['stage_folder']:
        spec['stage_path'] = ypath_join(issue_path, stage['stage_folder'])
    if 'previous_folder' in stage and stage['previous_folder']:
        spec['previous_path'] = ypath_join(issue_path, stage['previous_folder'])
    if 'execute_folder' in stage and stage['execute_folder']:
        spec['execute_path'] = ypath_join(issue_path, stage['execute_folder'])

    if 'inputs' in spec:
        inputs = spec['inputs']
        for table in inputs:
            if isinstance(inputs[table], list):
                inputs[table] = ypath_join(*[
                    jinja2.Template(part, undefined=jinja2.StrictUndefined).render(spec)
                    for part in inputs[table]
                ])
            else:
                inputs[table] = jinja2.Template(inputs[table], undefined=jinja2.StrictUndefined).render(spec)

    if 'outputs' in spec:
        for table, info in iteritems(spec['outputs']):
            info['path'] = ypath_join(spec['stage_path'], table)

    return spec


def validate_config(config):
    validate_config_schema(config)
    validate_config_operations(config)
    validate_config_inputs(config)
    validate_config_zones(config)
    validate_config_ribbons(config)
    validate_config_hidden(config)
    validate_config_task_types(config)
    validate_allowed_previous_states(config)


class StateNotFound(Exception):
    pass


class OperationNotFound(Exception):
    pass


class PathNotFound(Exception):
    pass


class QueueNotFound(Exception):
    pass


class StageNotFound(Exception):
    pass


class TaskTypeNotFound(Exception):
    pass


def validate_config_schema(config):
    import json
    import jsonschema

    schema = json.loads(fs.read_file('config_schema.json'))
    jsonschema.validate(instance=config, schema=schema)


def validate_allowed_previous_states(config):
    states = set()
    for stage_name in config['stages']:
        states.add(stage_name)
        if 'next_states' not in config['stages'][stage_name]:
            continue

        for state in config['stages'][stage_name]['next_states']:
            states.add(config['stages'][stage_name]['next_states'][state])

    for stage_name in config['stages']:
        if 'allowed_previous_states' not in config['stages'][stage_name]:
            continue

        for state in config['stages'][stage_name]['allowed_previous_states']:
            if state not in states:
                raise StateNotFound('Allowed previous state {} not found in stages or next_states'.format(
                    state
                ))


def validate_config_operations(config):
    for stage_name in config['stages']:
        if 'operations' not in config['stages'][stage_name]:
            continue

        for operation_name in config['stages'][stage_name]['operations']:
            if operation_name not in config['operations']:
                raise OperationNotFound('Operation {} found in config[stages], but not in config[operations]'.format(
                    operation_name
                ))


def validate_config_zones(config):
    for zone_name in config['zones']:
        queue_name = config['zones'][zone_name]['queue']
        if queue_name not in config['queues']:
            raise QueueNotFound('Queue {} found in config[zones], but not in config[queues]'.format(
                queue_name
            ))


def validate_config_ribbons(config):
    for ribbon_name in config['ribbons']:
        for stage_name in config['ribbons'][ribbon_name]:
            if stage_name not in config['stages']:
                raise StageNotFound('Stage {} found in config[ribbons], but not in config[stages]'.format(
                    stage_name
                ))


def validate_config_hidden(config):
    for operation_name in config['operations']:
        if 'outputs' not in config['operations'][operation_name]:
            continue

        for output_name in config['operations'][operation_name]['outputs']:
            output = config['operations'][operation_name]['outputs'][output_name]
            if 'hidden_in_stages' not in output:
                continue

            for stage_name in output['hidden_in_stages']:
                if stage_name not in config['stages']:
                    raise StageNotFound('Stage {} found in config[ribbons], but not in config[stages]'.format(
                        stage_name
                    ))


def validate_config_inputs(config):
    for stage_name in config['stages']:
        stage = config['stages'][stage_name]
        if 'operations' not in stage:
            continue

        for operation_name in stage['operations']:
            operation = config['operations'][operation_name]
            if 'inputs' not in operation:
                continue

            for input_name in operation['inputs']:
                input = operation['inputs'][input_name]
                if isinstance(input, list):
                    for part in input:
                        validate_config_input(stage, stage_name, operation_name, part)
                else:
                    validate_config_input(stage, stage_name, operation_name, input)


def validate_config_task_types(config):
    for stage_name in config['stages']:
        stage = config['stages'][stage_name]
        if 'performer' not in stage:
            continue

        if stage['performer'] not in config['task_types']:
            raise TaskTypeNotFound('Task type {} is in config[stages] but not in config[task_types]'.format(
                stage['performer']
            ))


def contains_whole_word(st, word):
    import re

    return re.search(r"\b" + re.escape(word) + r"\b", st)


def validate_config_input(stage, stage_name, operation_name, path):
    if contains_whole_word(path, 'stage_path'):
        if 'stage_folder' not in stage:
            raise PathNotFound('Path {} requires stage_folder in stage {} for operation {}'.format(
                path, stage_name, operation_name
            ))
    if contains_whole_word(path, 'previous_path'):
        if 'previous_folder' not in stage:
            raise PathNotFound('Path {} requires previous_folder in stage {} for operation {}'.format(
                path, stage_name, operation_name
            ))
    if contains_whole_word(path, 'execute_path'):
        if 'execute_folder' not in stage:
            raise PathNotFound('Path {} requires execute_folder in stage {} for operation {}'.format(
                path, stage_name, operation_name
            ))
