from __future__ import unicode_literals

import logging
import jinja2
from sandbox.common import fs
from sandbox.projects.yabs.partner_share.lib.create_tables.create_tables import create_filter_tables
from sandbox.projects.yabs.partner_share.lib.fast_changes.filter_schema.filter_schema import validate_filter_schema

logger = logging.getLogger(__name__)

FILTER_TEMPLATE_PATH = 'sandbox/projects/yabs/partner_share/lib/fast_changes/jinja/filter.jinja'
CHANGES_TEMPLATE_PATH = 'sandbox/projects/yabs/partner_share/lib/fast_changes/jinja/changes.jinja'
FILTER_IMPS_TEMPLATE_PATH = 'sandbox/projects/yabs/partner_share/lib/fast_changes/jinja/filter_imps.jinja'
CHANGES_IMPS_TEMPLATE_PATH = 'sandbox/projects/yabs/partner_share/lib/fast_changes/jinja/changes_imps.jinja'

MAX_PARTNER_SHARE_WITH_APPROVE_1 = 649999


def coalesce(tested_value, default_value):
    if tested_value is None:
        return default_value
    else:
        return tested_value


def update_conditions(filters):
    for filter in filters:
        for condition in filter['conditions']:
            condition['expr'] = condition['field']
            if condition['field'] in ['Segment', 'TrafficType', 'Login', 'Domain']:
                condition['value'] = ','.join(["'{v}'".format(v=v.strip()) for v in condition['value'].split(',')])
            if condition['field'] == 'PageID-ImpId':
                condition['expr'] = '(PageID,ImpID)'
                condition['value'] = ','.join(['(' + v.replace('-', ',') + ')' for v in condition['value'].split(',')])
            if condition['field'] == 'Domain':
                condition['expr'] = "splitByString(',', COALESCE(DomainList, ''))[1]"
            if condition['field'] == 'Segment':
                condition['expr'] = 'segment'
            if condition['field'] == 'TrafficType':
                condition['expr'] = 'traffic_type'
            if condition['field'] == 'IsInternal':
                condition['expr'] = '(OptionsYandexApp or OptionsYandexGroup or OptionsYandexPage)'


def _append_where_in(key, condition, where):
    if where[key]:
        where[key] += ' AND\n'
    where[key] += '        {expr} {operation} ({value})'.format(
        expr=condition['expr'],
        operation=condition.get('operation', 'in'),
        value=condition['value']
    )


def _append_where(key, expr, min, max, where):
    if where[key]:
        where[key] += ' AND\n'
    where[key] += '        {expr} >= {min} AND {expr} <= {max}'.format(
        expr=expr,
        min=min,
        max=max,
    )


def _build_where(filter):
    where = {}
    where['page_dsp_pages'] = ''
    where['page_dsp'] = ''
    where['segmentation'] = ''
    where['partner_page'] = ''
    where['page'] = ''
    for condition in filter['conditions']:
        if condition['field'] in ('DSPID', 'PageID'):
            _append_where_in('page_dsp_pages', condition, where)
        if condition['field'] in ('DSPID', 'PageID', 'PageID-ImpId'):
            _append_where_in('page_dsp', condition, where)
        if condition['field'] in ('Segment', 'TrafficType'):
            _append_where_in('segmentation', condition, where)
        if condition['field'] in ('Login', 'Domain', 'PartnerID'):
            _append_where_in('partner_page', condition, where)
        if condition['field'] in ('IsInternal'):
            _append_where_in('page', condition, where)

    _append_where_in('page_dsp_pages', dict(
        expr='DSPID',
        operation='not in',
        value='5,10'
    ), where)
    _append_where_in('page_dsp', dict(
        expr='DSPID',
        operation='not in',
        value='5,10'
    ), where)
    _append_where('page_dsp', 'PartnerShare', filter['CurrentPartnerShareFrom'], filter['CurrentPartnerShareTo'], where)
    return where


def generate_changes_chyt_query(chyt_cluster, filters, result_dir, ignore_partner_share_above):
    filter_template_txt = fs.read_file(FILTER_TEMPLATE_PATH).decode('utf-8')
    changes_template_txt = fs.read_file(CHANGES_TEMPLATE_PATH).decode('utf-8')
    filter_template = jinja2.Template(filter_template_txt, undefined=jinja2.StrictUndefined)
    changes_template = jinja2.Template(changes_template_txt, undefined=jinja2.StrictUndefined)

    filter_query = ''
    filter_id = 0
    logging.debug('Filters: {}'.format(str(filters)))
    for filter in filters:
        filter_id += 1
        where = _build_where(filter)
        filter_query += filter_template.render(
            EnableConsistentPageCollapse=not any(cond['field'] == 'PageID-ImpId' for cond in filter['conditions']),
            WherePageDSP=where['page_dsp'],
            WherePageDSPPages=where['page_dsp_pages'],
            WherePartnerPage=where['partner_page'],
            WherePage=where['page'],
            WhereSegmentation=where['segmentation'],
            CurrentPartnerShareFrom=filter['CurrentPartnerShareFrom'],
            CurrentPartnerShareTo=filter['CurrentPartnerShareTo'],
            NewPartnerShare=coalesce(filter['NewPartnerShare'], 'null'),
            DeltaPartnerShare=coalesce(filter['DeltaPartnerShare'], 'null'),
            PlannedPartnerShareExpr='PartnerShare + DeltaPartnerShare' if filter['NewPartnerShare'] is None else 'NewPartnerShare',
            ChangesFilterId=filter_id,
            ResultDir=result_dir,
        ) + '\n\n'

    return changes_template.render(
        ChytCluster=chyt_cluster,
        FilterQuery=filter_query,
        Filters=filters,
        ResultDir=result_dir,
        IgnorePartnerShareAbove=ignore_partner_share_above,
    )


def generate_changes_imps_chyt_query(chyt_cluster, filters, result_dir, ignore_partner_share_above, turnover_table_path):
    filter_imps_template_txt = fs.read_file(FILTER_IMPS_TEMPLATE_PATH).decode('utf-8')
    changes_imps_template_txt = fs.read_file(CHANGES_IMPS_TEMPLATE_PATH).decode('utf-8')
    filter_template = jinja2.Template(filter_imps_template_txt, undefined=jinja2.StrictUndefined)
    changes_template = jinja2.Template(changes_imps_template_txt, undefined=jinja2.StrictUndefined)

    filter_queries = []
    filter_id = 0
    for filter in filters:
        filter_id += 1
        where = _build_where(filter)
        filter_queries.append(filter_template.render(
            WherePageDSP=where['page_dsp'],
            WherePageDSPPages=where['page_dsp_pages'],
            WherePartnerPage=where['partner_page'],
            WherePage=where['page'],
            WhereSegmentation=where['segmentation'],
            CurrentPartnerShareFrom=filter['CurrentPartnerShareFrom'],
            CurrentPartnerShareTo=filter['CurrentPartnerShareTo'],
            NewPartnerShare=coalesce(filter['NewPartnerShare'], 'null'),
            DeltaPartnerShare=coalesce(filter['DeltaPartnerShare'], 'null'),
            PlannedPartnerShareExpr='PartnerShare + DeltaPartnerShare' if filter['NewPartnerShare'] is None else 'NewPartnerShare',
            ChangesFilterId=filter_id,
            TurnoverTablePath=turnover_table_path,
            ResultDir=result_dir,
        ))

    return changes_template.render(
        ChytCluster=chyt_cluster,
        FilterQueries=filter_queries,
        ResultDir=result_dir,
        TurnoverTablePath=turnover_table_path,
        IgnorePartnerShareAbove=ignore_partner_share_above,
    )


def prepare_changes(spec):
    from yql.api.v1.client import YqlClient
    from yql.client import operation
    from yt.wrapper import YtClient

    validate_filter_schema(spec['filters'])

    logging.debug('Rendering query')

    update_conditions(spec['filters']['filters'])

    query = generate_changes_chyt_query(
        chyt_cluster=spec['chyt_cluster'],
        filters=spec['filters']['filters'],
        result_dir=spec['stage_path'],
        ignore_partner_share_above=spec.get('ignore_partner_share_above', 1000000),
    )
    logging.debug('Changes CHYT query: {}'.format(query))

    yt = YtClient(
        token=spec['yql_token'],
        proxy=spec['yt_cluster'],
    )

    turnover_table_path = yt.get('//home/yabs/tac-manager/turnover/last&/@target_path')

    query_imps = generate_changes_imps_chyt_query(
        chyt_cluster=spec['chyt_cluster'],
        filters=spec['filters']['filters'],
        result_dir=spec['stage_path'],
        ignore_partner_share_above=spec.get('ignore_partner_share_above', 1000000),
        turnover_table_path=turnover_table_path,
    )
    logger.debug('Changes_imps CHYT query: {}'.format(query))

    logger.debug('Filter changes CHYT query: %s', query)

    if spec.get('dont_run_yql'):
        return

    logging.debug('Creating tables')
    create_filter_tables(
        yt_cluster=spec['yt_cluster'],
        yt_token=spec['yql_token'],
        filters=spec['filters']['filters'],
        result_dir=spec['stage_path'],
    )

    yql_client = YqlClient(token=spec['yql_token'])

    request = yql_client.query(query, syntax_version=1, title='PartnerShare changes generation in YQL')
    request.type = operation.YqlOperationType.CLICKHOUSE
    request.run()
    spec['task'].set_info('Filter changes CHYT operation: <a href={url}>{url}</a>'.format(
        url=request.share_url
    ), do_escape=False)

    request_imps = yql_client.query(query_imps, syntax_version=1, title='PartnerShare changes_imps generation in YQL')
    request_imps.type = operation.YqlOperationType.CLICKHOUSE
    request_imps.run()
    spec['task'].set_info('Filter changes_imps CHYT operation: <a href={url}>{url}</a>'.format(
        url=request_imps.share_url
    ), do_escape=False)

    results = request.get_results()
    results_imps = request_imps.get_results()

    if not results.is_success:
        msg = '\n'.join([str(err) for err in results.errors])
        raise RuntimeError('Error when executing YQL query: %s' % msg)

    if not results_imps.is_success:
        msg = '\n'.join([str(err) for err in results_imps.errors])
        raise RuntimeError('Error when executing YQL query: %s' % msg)
