import json
import logging
from sandbox.projects.yabs.partner_share.lib.utils import retry


logger = logging.getLogger(__name__)


def get_sandbox_link(id):
    return "https://sandbox.yandex-team.ru/task/{}".format(id)


def get_queue_from_issue(issue):
    return issue[:issue.rfind('-')]


class FileNotFound(Exception):
    pass


class StartrekHelper():
    def __init__(self, useragent, startrek_api_url, st_token, local_fields_prefix):
        from startrek_client import Startrek

        self.st_client = Startrek(
            useragent=useragent,
            base_url=startrek_api_url,
            token=st_token
        )
        self.local_fields_prefix = local_fields_prefix

    def get_local_name(self, name):
        return self.local_fields_prefix + "--" + name

    def get_all_tickets(self, queue, tacman_statuses):
        issues = self.st_client.issues.find(
            filter={
                "queue": queue,
                self.get_local_name("tacman_status"): tacman_statuses
            }
        )
        return [issue.key for issue in issues]

    def get_recent_tickets(self, queue, tacman_statuses, updated_from):
        issues = self.st_client.issues.find(
            filter={
                "queue": queue,
                self.get_local_name("tacman_status"): tacman_statuses,
                "updated": {
                    "from": updated_from
                },
            }
        )
        return [issue.key for issue in issues]

    @retry()
    def change_local_field(self, ticket, local_field_name, value):
        issue = self.st_client.issues[ticket]
        issue.update(
            **{
                self.get_local_name(local_field_name): value
            }
        )

    def get_local_field(self, ticket, local_field_name):
        issue = self.st_client.issues[ticket]
        logger.info('Get %s field for ticket %s' % (self.get_local_name(local_field_name), ticket))
        return issue[self.get_local_name(local_field_name)]

    def get_field(self, ticket, field_name):
        issue = self.st_client.issues[ticket]
        return issue[field_name]

    def add_comment(self, ticket, text):
        return self.st_client.issues[ticket].comments.create(text=text)

    def get_attachment(self, ticket, file_name):
        issue = self.st_client.issues[ticket]
        logger.debug('Get issue name {}: {}'.format(ticket, issue))
        attachments = issue.attachments.get_all()
        attachments = sorted(attachments, key=lambda attachment: attachment["createdAt"], reverse=True)
        for attachment in attachments:
            if attachment["name"] == file_name:
                attachment.download_to(".")
                with open(file_name, "r") as f:
                    return json.load(f)
        raise FileNotFound()

    def get_followers(self, ticket):
        return [
            u.id
            for u in self.st_client.issues[ticket].followers
        ]

    def set_followers(self, ticket, followers):
        return self.st_client.issues[ticket].update(followers=followers)
