import time
import subprocess
from functools import wraps
import logging

logger = logging.getLogger(__name__)


class Color:
    RED = '\033[31m'
    DARKGREEN = '\033[32m'
    DARKYELLOW = '\033[33m'
    DARKCYAN = '\033[36m'
    WHITE = '\033[37m'
    GRAY = '\033[90m'
    PINK = '\033[91m'
    GREEN = '\033[92m'
    YELLOW = '\033[93m'
    BLUE = '\033[94m'
    PURPLE = '\033[95m'
    CYAN = '\033[96m'
    BRIGHT = '\033[97m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'
    BLINK = '\033[5m'
    END = '\033[0m'


def retry(
    exceptions=Exception,
    tries=4,
    delay=1,
    backoff_multiplier=2,
):
    """Retry calling the decorated function using an exponential backoff_multiplier.
    http://www.saltycrane.com/blog/2009/11/trying-out-retry-decorator-python/
    original from: http://wiki.python.org/moin/PythonDecoratorLibrary#Retry
    exceptions - the exception to check. may be a tuple of exceptions to check
                 type: Exception or tuple
    tries - number of times to try (not retry) before giving up
            type: int
    delay - initial delay between retries in seconds
            type: int
    backoff_multiplier - backoff multiplier e.g. value of 2 will double the delay each retry
                         type: int
    """
    def deco_retry(f):
        @wraps(f)
        def f_retry(*args, **kwargs):
            retries, retry_interval = tries, delay
            while retries > 0:
                try:
                    return f(*args, **kwargs)
                except exceptions as ex:
                    retries -= 1
                    if retries > 0:
                        logger.warning("{}, Retrying in {} seconds...".format(
                            repr(ex),
                            int(retry_interval)
                        ))
                        time.sleep(retry_interval)
                        retry_interval *= backoff_multiplier
                    else:
                        raise
        return f_retry
    return deco_retry


def run(command, shell=True):
    try:
        res = subprocess.check_output(
            command,
            shell=shell,
        ).decode("utf-8")
    except subprocess.CalledProcessError as e:
        return e.output.decode("utf-8")
    return res
