import jinja2
import logging

from sandbox.common import fs
from sandbox.projects.yabs.partner_share.lib.utils import retry

logger = logging.getLogger(__name__)


def run_yql_query(spec, yql_query):
    if spec.get('dont_run_yql'):
        return

    from yql.api.v1.client import YqlClient
    from yql.client import operation

    yql_client = YqlClient(token=spec['yql_token'])
    logger.debug('{}: {}'.format(spec['title'], yql_query))

    request = yql_client.query(yql_query, syntax_version=1, title=spec['title'] + ' YQL')
    if spec.get('clickhouse'):
        request.type = operation.YqlOperationType.CLICKHOUSE
    request.run()
    logger.debug('{} operation: {}'.format(spec['title'], request.share_url))

    spec['task'].set_info('{title}: <a href={url}>{url}</a>'.format(
        title=spec['title'],
        url=request.share_url
    ), do_escape=False)

    results = request.get_results()
    logger.debug('{} operation result: {}'.format(spec['title'], results.is_success))
    if not results.is_success:
        msg = '\n'.join([str(err) for err in results.errors])
        raise Exception('Error when executing {} YQL query: {}'.format(spec['title'], msg))

    return request.share_url


@retry(tries=3, delay=1, backoff_multiplier=2)
def retry_yql_query(spec, yql_query):
    return run_yql_query(spec, yql_query)


def yql_query(spec):
    query_template_txt = fs.read_file(spec['query_template']).decode('utf-8')
    query_template = jinja2.Template(query_template_txt, undefined=jinja2.StrictUndefined)
    yql_query = query_template.render(spec)

    if spec.get('clickhouse'):
        return retry_yql_query(spec, yql_query)
    else:
        return run_yql_query(spec, yql_query)
