from __future__ import unicode_literals
from __future__ import print_function

import os
import json
import argparse
from time import sleep
from sandbox.projects.yabs.partner_share.lib.config.config import (
    get_config,
    get_daemons,
)
from sandbox.projects.yabs.partner_share.lib.utils import (
    Color,
    run,
)
from sandbox.projects.yabs.partner_share.lib.sandbox_helper import (
    SandboxHelper,
    get_sandbox_link,
)
from sandbox.projects.yabs.partner_share.lib.control_helper import ControlHelper
import logging

logging.basicConfig(level=logging.WARN, format='%(asctime)s %(message)s')
l = logging.getLogger(__name__)
config = get_config()

OWNER = 'YABS_SERVER_SANDBOX_TESTS'


def get_st_token():
    token = ''
    token_path = os.path.expanduser("~/.startrek/token")
    if os.path.exists(token_path):
        with open(token_path, 'r') as fil:
            token = fil.read().replace('\n', '')
            if token:
                print('Using ' + token_path + ' file for Startrek token')
    else:
        raise RuntimeError(
            'Please put startrek token to ~/.startrek/token from '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=5f671d781aca402ab7460fde4050267b'
        )
    return token


def get_ok_token():
    token = ''
    token_path = os.path.expanduser("~/.ok/token")
    if os.path.exists(token_path):
        with open(token_path, 'r') as fil:
            token = fil.read().replace('\n', '')
            if token:
                print('Using ' + token_path + ' file for OK token')
    else:
        raise RuntimeError(
            'Please put OK token to ~/.ok/token from '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=0137c50bcf1e45afaa7a9fdd8d17fc72'
        )
    return token


def get_sandbox_token():
    token = ''
    token_path = os.path.expanduser("~/.sandbox/token")
    if os.path.exists(token_path):
        with open(token_path, 'r') as fil:
            token = fil.read().replace('\n', '')
            if token:
                print('Using ' + token_path + ' file for sandbox token')
    else:
        raise RuntimeError(
            'Please put Sandbox token to ~/.sandbox/token from '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=968e0e6d85d647feb327d893a42cf26f&optional_scope=sandbox:api'
        )
    return token


def get_yql_token():
    token = os.environ.get('YQL_TOKEN')
    if token:
        print('Using YQL_TOKEN environment variable for YQL token')
    if not token:
        token = os.environ.get('YT_TOKEN')
        if token:
            print('Using YT_TOKEN environment variable for YQL token')
    if not token:
        token_path = os.path.expanduser("~/.yt/token")
        if os.path.exists(token_path):
            with open(token_path, 'r') as fil:
                token = fil.read().replace('\n', '')
                if token:
                    print('Using ' + token_path + ' file for YQL token')
    if not token:
        raise RuntimeError(
            'Cannot find YQL token in environment variables YT_TOKEN or YQL_TOKEN or in file ~/.yt/token '
            'Please get it from https://yql.yandex-team.ru'
        )
    return token


def show_tasks(task_type, queue, sandbox_helper, limit=12):
    print(task_type, queue, 'tasks:')
    tasks = sandbox_helper.get_tasks(task_type, queue, limit=100)['items']
    for task in tasks[:limit]:
        color = Color.WHITE
        if task['status'] in ('SUCCESS'):
            color = Color.GREEN
        if task['status'] in ('ENQUEUED'):
            color = Color.WHITE
        if task['status'] == 'DRAFT':
            color = Color.GRAY
        if task['status'] in ('EXCEPTION', 'FAILURE', 'STOPPED'):
            color = Color.RED
        if task['status'] == 'EXECUTING':
            color = Color.BLUE
        print(
            get_sandbox_link(task['id']),
            task['time']['updated'],
            task['requirements']['tasks_resource'],
            color + task['status'] + Color.END
        )
    return tasks


def wait_tasks(tasks, sandbox_helper):
    for _ in range(100):
        for task_id in tasks:
            status = sandbox_helper.get_task(task_id)['status']
            print('Task {} is {}'.format(task_id, status))
            if status in ('ASSIGNED', 'EXECUTING', 'EXCEPTION', 'STOPPED'):
                tasks.remove(task_id)

        if not tasks:
            break

        sleep(3)


def replace_task(task_type, queue, control_helper, tasks, sandbox_helper):
    seen_resources = set(t['requirements']['tasks_resource'] for t in tasks)

    res = json.loads(run('bin/partner_share_tasks run --enable-taskbox --create-only --owner {owner} --tid {task_id}'.format(
        owner=OWNER,
        task_id=tasks[0]['id'],
    )))
    task_id = res['task']['id']
    resource_id = res['resource']['id']

    control_helper.set_resource_id(queue, task_type, resource_id)

    if resource_id in seen_resources:
        print(Color.YELLOW + 'Will not start DRAFT task with same resource' + Color.END)
    else:
        print(Color.YELLOW + 'Starting task' + Color.END)
        sandbox_helper.start_task(task_id)
        return task_id


def main():
    parser = argparse.ArgumentParser(description="TacMan integration testing")
    queues = [q for q in config['queues']]
    parser.add_argument(
        "command",
        help="Select your command",
        choices=['info', 'deploy'],
    )
    parser.add_argument(
        "queue",
        help="Choose startrek queue",
        choices=queues,
    )
    args = parser.parse_args()

    sandbox_helper = SandboxHelper(get_sandbox_token())

    if args.command == 'info':
        for task_type in get_daemons(config):
            tasks = show_tasks(task_type, args.queue, sandbox_helper)
        return

    control_helper = ControlHelper(get_yql_token())
    started_tasks = []
    for task_type in get_daemons(config):
        tasks = show_tasks(task_type, args.queue, sandbox_helper)

        task_id = replace_task(task_type, args.queue, control_helper, tasks, sandbox_helper)
        if task_id:
            started_tasks.append(task_id)

    wait_tasks(started_tasks, sandbox_helper)
    for task_type in get_daemons(config):
        tasks = show_tasks(task_type, args.queue, sandbox_helper, limit=6)


if __name__ == "__main__":
    main()
