import logging
import time
import datetime
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.yabs.partner_share.lib.st_helper import StartrekHelper, get_sandbox_link
from sandbox.projects.yabs.partner_share.lib.control_helper import ControlHelper
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.yabs.partner_share.lib.config.config import get_config, render_spec

from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus

TESTING_STARTREK_API_ENDPOINT = 'https://st-api.test.yandex-team.ru'
STARTREK_API_ENDPOINT = 'https://st-api.yandex-team.ru'

TACMAN_YT_CLUSTER = 'hahn'
TACMAN_CHYT_CLUSTER = 'chyt.hahn/lelby_ckique'

TACMAN_YT_LOCK_CLUSTER = 'hahn'

ISSUE_UPDATE_SECONDS = 10

logger = logging.getLogger(__name__)


class StopTaskGracefully(Exception):
    pass


class TacmanFilterServer(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 512
        environments = [
            PipEnvironment('yql', version='1.2.91', use_wheel=True),
            PipEnvironment('yandex-yt', version='0.9.17', use_wheel=True),
            PipEnvironment('startrek_client', version='2.5.0', use_wheel=True),
            PipEnvironment('jsonschema', version='3.0.2', use_wheel=True)
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True
        tokens = sdk2.parameters.YavSecret(
            'Yav secret with robot tokens',
            default='sec-01fnrae4z6bj38bfjdpxgrjggm'
        )
        use_testing_startrek = sdk2.parameters.Bool('Use testing startrek')
        st_queue = sdk2.parameters.String(
            'Startrek queue to process',
            default='TESTTACCHANGES',
            hint=True,
        )
        repeats = sdk2.parameters.Integer(
            'Repeat checking tickets',
            default=1
        )
        st_period = sdk2.parameters.Integer(
            'Check startrek queue once every Nth cycle',
            default=1
        )
        sleep_between_repeats = sdk2.parameters.Float(
            'Number of seconds to sleep between checking tickets',
            default=1.0)
        auto_restart = sdk2.parameters.Bool(
            'Restart task automatically on success',
            default=False
        )
        auto_restart_on_failure = sdk2.parameters.Bool(
            'Restart task automatically on failure',
            default=False
        )
        filter_json_startrek_file = sdk2.parameters.String(
            "Filter json file name in startrek",
            default="filter.json",
            required=True
        )

        search_released_resource = sdk2.parameters.Bool(
            "Search for Released Resource",
            default=False
        )

        release_version = sdk2.parameters.String(
            "Release version",
            choices=[(v, v) for _, v in ReleaseStatus.iteritems()],
            default=ReleaseStatus.TESTING
        )

    def on_create(self):
        if not self.Parameters.search_released_resource:
            return

        tasks_resource_testing = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'name': 'TacmanBinaryResource', 'released': 'testing'},
            state=[State.READY],
        ).first()

        tasks_resource_stable = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'name': 'TacmanBinaryResource', 'released': 'stable'},
            state=[State.READY],
        ).first()

        if self.Parameters.release_version == ReleaseStatus.TESTING:
            if not tasks_resource_testing or tasks_resource_stable.major_release_num < tasks_resource_testing.major_release_num:
                self.Requirements.tasks_resource = tasks_resource_testing
                return

        self.Requirements.tasks_resource = tasks_resource_stable

    def filter(self, ticket):
        from sandbox.projects.yabs.partner_share.lib.fast_changes.fast_changes import (
            prepare_changes
        )

        spec = render_spec(
            config=self.config,
            stage_name='FILTER',
            operation_name='prepare_changes',
            task=self,

            yt_cluster=TACMAN_YT_CLUSTER,
            chyt_cluster=TACMAN_CHYT_CLUSTER,
            yql_token=self.Parameters.tokens.data()['yql_token'],
            yt_root='//',

            issue=ticket,
            direct_issue=ticket,
            filters=self.st_helper.get_attachment(
                ticket,
                self.Parameters.filter_json_startrek_file
            ),
        )

        prepare_changes(spec)

    def process_ticket(self, ticket):
        logging.debug('Processing ticket {}'.format(ticket))

        self.set_info(
            '{ticket}: {process} started'.format(
                process=self.stage['next_states']['process'],
                ticket=ticket,
            )
        )
        logging.debug('Setting {} field of ticket {} to my task id {}'.format(
            self.stage['task_id_field'],
            ticket,
            self.id
        ))
        self.st_helper.change_local_field(
            ticket=ticket,
            local_field_name=self.stage['task_id_field'],
            value=self.id
        )

        logging.debug('Setting tacman_status of {} to {}'.format(ticket, self.stage['next_states']['process']))
        self.st_helper.change_local_field(
            ticket=ticket,
            local_field_name='tacman_status',
            value=self.stage['next_states']['process']
        )
        self.last_issue_update[ticket] = time.time()

        self.current_ticket = ticket

        self.filter(ticket)

        logging.debug('Setting tacman_status of {} to {}'.format(ticket, self.stage['next_states']['success']))
        self.st_helper.change_local_field(
            ticket=ticket,
            local_field_name='tacman_status',
            value=self.stage['next_states']['success']
        )
        self.current_ticket = None

        self.set_info(
            '{ticket}: {state}'.format(
                ticket=ticket,
                state=self.stage['next_states']['success']
            )
        )

        self.st_helper.add_comment(ticket, '{process} -> {success}: {my_url}'.format(
            process=self.stage['next_states']['process'],
            success=self.stage['next_states']['success'],
            my_url=get_sandbox_link(self.id)
        ))

    def detect_ticket_hint(self):
        ticket_hint = self.control_helper.get_issue_hint(
            self.Parameters.st_queue,
            self.type,
        )
        if ticket_hint:
            logging.debug('Detected hint {}'.format(ticket_hint))
            self.control_helper.set_issue_hint(
                self.Parameters.st_queue,
                self.type,
                ''
            )
        return ticket_hint

    def find_tickets(self, cycle):
        from_date = datetime.date.today() - datetime.timedelta(days=1)
        logging.debug('Searching from date {}'.format(from_date))

        if cycle % self.Parameters.st_period == 0:
            # Find tickets in needed status ('FILTER') - this has about 10 seconds delay
            tickets = self.st_helper.get_all_tickets(
                queue=self.Parameters.st_queue,
                tacman_statuses=['FILTER'],
            )
        else:
            tickets = []

        # Detect hint to changed ticket in first ticket of the queue, as it works much faster than search
        ticket_hint = self.detect_ticket_hint()
        if ticket_hint:
            tickets.append(ticket_hint)

        for ticket in tickets:
            if self.st_helper.get_local_field(ticket, 'tacman_status') != 'FILTER':
                continue

            # Ignore ticket if we changed it not long ago because it's status can lag on replica
            if ticket in self.last_issue_update:
                if time.time() - self.last_issue_update[ticket] < ISSUE_UPDATE_SECONDS:
                    logging.debug('Ticket {} ignored during udpate in API'.format(ticket))
                    continue

            self.process_ticket(ticket)

    def show_started_task(self, title, task_id):
        self.set_info('Started <a href={url}>"{title}" task</a>'.format(
            title=title,
            url=get_sandbox_link(task_id),
        ), do_escape=False)

    def enque_me(self):
        task = TacmanFilterServer(
            self,
            description='Auto restarted filter server task ' + self.Parameters.st_queue,
            owner=self.owner,
            tokens=self.Parameters.tokens,
            use_testing_startrek=self.Parameters.use_testing_startrek,
            st_queue=self.Parameters.st_queue,
            repeats=self.Parameters.repeats,
            sleep_between_repeats=self.Parameters.sleep_between_repeats,
            auto_restart=self.Parameters.auto_restart,
            auto_restart_on_failure=self.Parameters.auto_restart_on_failure,
            filter_json_startrek_file=self.Parameters.filter_json_startrek_file,
        )
        task.enqueue()
        self.show_started_task('Auto restarted filter server task ' + self.Parameters.st_queue, task.id)

    def run_server(self):
        with self.memoize_stage.process_tickets(commit_on_entrance=True):
            self.set_info('Started listening to ticket changes')
            for cycle in range(self.Parameters.repeats):
                self.find_tickets(cycle)
                if cycle < self.Parameters.repeats - 1:
                    time.sleep(self.Parameters.sleep_between_repeats)

        if self.Parameters.auto_restart:
            self.enque_me()

    def on_enqueue(self):
        if self.Parameters.st_queue == 'TESTTACCHANGES':
            semaphore_name = 'TACMAN_FILTER_SERVER_TEST'
        else:
            semaphore_name = 'TACMAN_FILTER_SERVER'
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name=semaphore_name)
            ]
        )

    def on_execute(self):
        self.current_ticket = None
        self.last_issue_update = {}
        self.config = get_config()
        self.stage = self.config['stages']['FILTER']

        tokens = self.Parameters.tokens.data()
        self.st_helper = StartrekHelper(
            useragent='sandbox',
            startrek_api_url=TESTING_STARTREK_API_ENDPOINT
                if self.Parameters.use_testing_startrek
                else STARTREK_API_ENDPOINT,
            st_token=tokens['st_token'],
            local_fields_prefix=self.config['queues'][self.Parameters.st_queue]['local_fields_prefix']
        )
        self.control_helper = ControlHelper(tokens['yql_token'])

        try:
            self.run_server()
        except StopTaskGracefully as e:
            self.set_info('Stopping this task due to received command: {}'.format(str(e)))

    def set_issue_status(self, status):
        self.set_info(
            "Setting {status} status for ticket <a href={url}>{ticket}</a>".format(
                status=status,
                url='https://st.yandex-team.ru/{}'.format(self.current_ticket),
                ticket=self.current_ticket,
            ),
            do_escape=False
        )
        self.st_helper.change_local_field(self.current_ticket, "tacman_status", status)
        self.st_helper.add_comment(self.current_ticket, '{status} status set by {sb_url}'.format(
            status=status,
            sb_url=get_sandbox_link(self.id),
        ))

    def finish(self, prev_status, status):
        self.config = get_config()

        if status == 'SUCCESS':
            return
        if self.Parameters.auto_restart_on_failure and status != 'STOPPED':
            self.enque_me()
        if not hasattr(self, 'current_ticket') or not self.current_ticket:
            return
        self.set_issue_status('FILTERED_FAIL')

    def on_finish(self, prev_status, status):
        self.finish(prev_status, status)

    def on_break(self, prev_status, status):
        self.finish(prev_status, status)

    def on_wait(self, prev_status, status):
        logging.debug('on_wait {}'.format(status))
