import logging
import time
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.yabs.partner_share.lib.st_helper import (
    StartrekHelper,
    get_sandbox_link
)
from sandbox.projects.yabs.partner_share.tasks.apply_changes import TacmanApplyChanges
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.yabs.partner_share.lib.control_helper import ControlHelper
from sandbox.projects.yabs.partner_share.lib.ok_helper import OkHelper, iframe_text
from sandbox.projects.yabs.partner_share.lib.config.config import (
    get_config,
    APPROVAL_TYPE_FIELD
)

from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus


TESTING_STARTREK_API_ENDPOINT = 'https://st-api.test.yandex-team.ru'
STARTREK_API_ENDPOINT = 'https://st-api.yandex-team.ru'

TACMAN_YT_CLUSTER = 'hahn'
CHANGES_TABLE = 'changes'
ISSUE_UPDATE_SECONDS = 10


class StopTaskGracefully(Exception):
    pass


class TacmanQueue(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 512
        environments = [
            PipEnvironment('yql', version='1.2.91', use_wheel=True),
            PipEnvironment('yandex-yt', version='0.9.17', use_wheel=True),
            PipEnvironment('startrek_client', version='2.5.0', use_wheel=True)
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True
        dry_run = sdk2.parameters.Bool(
            'Dry run',
            default=True
        )
        tokens = sdk2.parameters.YavSecret(
            'Yav secret with robot tokens',
            default='sec-01fnrae4z6bj38bfjdpxgrjggm'
        )
        use_testing_startrek = sdk2.parameters.Bool('Use testing startrek')
        st_queue = sdk2.parameters.String(
            'Startrek queue to process',
            default='TESTTACCHANGES',
            hint=True,
        )
        repeats = sdk2.parameters.Integer(
            'Repeat checking tickets for X times',
            default=1
        )
        st_period = sdk2.parameters.Integer(
            'Check startrek queue once every Nth cycle',
            default=1
        )
        sleep_between_repeats = sdk2.parameters.Integer(
            'Number of seconds to sleep between checking tickets',
            default=5)
        auto_restart = sdk2.parameters.Bool(
            'Restart task automatically on success',
            default=False
        )
        auto_restart_on_failure = sdk2.parameters.Bool(
            'Restart task automatically on failure',
            default=False
        )

        search_released_resource = sdk2.parameters.Bool(
            "Search for Released Resource",
            default=False
        )

        release_version = sdk2.parameters.String(
            "Release version",
            choices=[(v, v) for _, v in ReleaseStatus.iteritems()],
            default=ReleaseStatus.TESTING
        )

    def on_create(self):
        if not self.Parameters.search_released_resource:
            return

        tasks_resource_testing = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'name': 'TacmanBinaryResource', 'released': 'testing'},
            state=[State.READY],
        ).first()

        tasks_resource_stable = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'name': 'TacmanBinaryResource', 'released': 'stable'},
            state=[State.READY],
        ).first()

        if self.Parameters.release_version == ReleaseStatus.TESTING:
            if not tasks_resource_testing or tasks_resource_stable.major_release_num < tasks_resource_testing.major_release_num:
                self.Requirements.tasks_resource = tasks_resource_testing
                return

        self.Requirements.tasks_resource = tasks_resource_stable

    def add_user_param(self, params, ticket):
        params['user'] = self.st_helper.get_local_field(ticket, 'tacman_executor')

    def add_ignore_partner_share_above(self, params, ticket):
        from sandbox.projects.yabs.partner_share.lib.fast_changes.fast_changes import MAX_PARTNER_SHARE_WITH_APPROVE_1

        if self.st_helper.get_local_field(ticket, APPROVAL_TYPE_FIELD) == 1:
            params['ignore_partner_share_above'] = MAX_PARTNER_SHARE_WITH_APPROVE_1

    def add_direct_issue_param(self, params, ticket):
        params['direct_issue'] = self.st_helper.get_local_field(ticket, 'tacman_direct_issue')

    def validate_ticket_status(self, ticket, status, real_status):
        if status != real_status:
            return False

        # Ignore ticket if we changed it not long ago because it's status can lag on replica
        if ticket in self.last_issue_update:
            if time.time() - self.last_issue_update[ticket] < ISSUE_UPDATE_SECONDS:
                logging.debug('Ticket {} flap {} in API'.format(
                    ticket,
                    status
                ))
                return False
        return True

    def process_ticket(
        self,
        ticket,
        current_status,
        real_status,
    ):
        if not self.validate_ticket_status(ticket, current_status, real_status):
            return

        logging.debug(
            'Ticket in {} status: {}'.format(
                current_status,
                ticket
            )
        )

        stage = self.config['stages'][current_status]
        new_status = stage['next_states']['process']
        author = self.st_helper.get_field(ticket, 'createdBy').id
        uid = 1

        if current_status == 'EXECUTE' or current_status == 'REVERT':
            approve_status = self.ok_helper.get_approve_status(
                ticket=ticket,
                uid=uid,
                author=author
            )
            logging.debug(
                'Ticket {} approve status {}'.format(
                    ticket,
                    approve_status
                )
            )
            if approve_status != 'closed':
                self.set_info('{} not approved'.format(ticket))
                new_status = stage['next_states']['fail']
                self.st_helper.change_local_field(
                    ticket=ticket,
                    local_field_name='tacman_status',
                    value=new_status
                )
                return

        task_params = dict(
            description='{status} {issue} child of {id}'.format(
                status=new_status,
                issue=ticket,
                id=self.id
            ),
            owner=self.owner,
            tokens=self.Parameters.tokens,
            apply_new_tacman_status=True,
            issue=ticket,
            stage_name=current_status,
            use_testing_startrek=self.Parameters.use_testing_startrek,
        )
        self.add_user_param(task_params, ticket)
        if stage.get('require_direct_issue'):
            self.add_direct_issue_param(task_params, ticket)
        if stage.get('ignore_partner_share_above'):
            self.add_ignore_partner_share_above(task_params, ticket)
        if 'delay_next_stage_on_success' in stage:
            task_params['delay_on_success'] = stage['delay_next_stage_on_success']['delay_in_minutes']
            task_params['status_after_delay'] = stage['delay_next_stage_on_success']['next_stage']
        task = TacmanApplyChanges(self, **task_params)

        logging.debug('Setting {} field of ticket {} to new task id {}'.format(
            stage['task_id_field'],
            ticket,
            task.id
        ))
        self.st_helper.change_local_field(
            ticket=ticket,
            local_field_name=stage['task_id_field'],
            value=task.id
        )

        logging.debug('Setting tacman_status of {} to {}'.format(ticket, new_status))
        self.st_helper.change_local_field(
            ticket=ticket,
            local_field_name='tacman_status',
            value=new_status
        )
        task.enqueue()
        self.last_issue_update[ticket] = time.time()

        self.set_info(
            '{ticket}: {current_status} -> {new_status}: started <a href={sb_url}>{sb_url}</a>'.format(
                ticket=ticket,
                current_status=current_status,
                new_status=new_status,
                sb_url=get_sandbox_link(task.id),
            ),
            do_escape=False
        )
        self.st_helper.add_comment(ticket, '{current_status} -> {new_status}: started {sb_url}\nQueue processor: {my_url}'.format(
            current_status=current_status,
            new_status=new_status,
            sb_url=get_sandbox_link(task.id),
            my_url=get_sandbox_link(self.id)
        ))

    def process_approve(self, ticket, current_status, real_status):
        if not self.validate_ticket_status(ticket, current_status, real_status):
            return

        stage = self.config['stages'][current_status]
        new_status = None

        author = self.st_helper.get_field(ticket, 'createdBy').id
        summary = self.st_helper.get_field(ticket, 'summary')
        uid = 1
        logging.debug('Author %s, summary %s', author, summary)

        if current_status == 'APPROVE' or current_status == 'REVERT_APPROVE':
            approve_type = self.st_helper.get_local_field(ticket, APPROVAL_TYPE_FIELD)
            logging.info("Approveval Type {} in Ticket {}".format(approve_type, ticket))

            approve_config = self.config['approvals'][approve_type]

            result = self.ok_helper.start_approve(
                ticket=ticket,
                uid=uid,
                author=author,
                approve_type=approve_type,
                text=summary,
                admin_groups=[]
            )

            uuid = result.json().get('uuid')
            logging.debug('OK UUID: %s', uuid)
            if uuid:
                self.st_helper.change_local_field(
                    ticket=ticket,
                    local_field_name='ok_uuid',
                    value=uuid,
                )

            if 'followers' in approve_config:
                followers = self.st_helper.get_followers(ticket)
                logging.debug('Adding followers {} to {}'.format(
                    followers,
                    approve_config['followers'],
                ))
                followers.extend(approve_config['followers'])
                followers = self.st_helper.set_followers(ticket, list(set(followers)))

            self.st_helper.add_comment(ticket, iframe_text(
                ticket=ticket,
                uid=uid,
                author=author
            ))

            new_status = stage['next_states']['process']

        elif current_status == 'APPROVE_SUSPEND' or current_status == 'REVERT_APPROVE_SUSPEND':
            self.ok_helper.suspend_approve(
                ticket=ticket,
                uid=uid,
                author=author,
            )
            new_status = stage['next_states']['success']

        elif current_status == 'APPROVE_RESUME' or current_status == 'REVERT_APPROVE_RESUME':
            self.ok_helper.resume_approve(
                ticket=ticket,
                uid=uid,
                author=author,
            )
            new_status = stage['next_states']['success']

        if new_status:
            self.st_helper.change_local_field(
                ticket=ticket,
                local_field_name='tacman_status',
                value=new_status
            )
            self.st_helper.change_local_field(
                ticket=ticket,
                local_field_name=stage['task_id_field'],
                value=self.id
            )

            self.set_info(
                '{ticket}: {current_status} -> {new_status}'.format(
                    ticket=ticket,
                    current_status=current_status,
                    new_status=new_status,
                ),
            )
            self.last_issue_update[ticket] = time.time()
        else:
            self.set_info(
                "Current Status {current_status} in Ticket {ticket} can not be processed by method process_approve".format(
                    current_status=current_status,
                    ticket=ticket
                )
            )

    def detect_ticket_hint(self):
        ticket_hint = self.control_helper.get_issue_hint(
            self.Parameters.st_queue,
            self.type,
        )
        if ticket_hint:
            logging.debug('Detected hint {}'.format(ticket_hint))
            self.control_helper.set_issue_hint(
                self.Parameters.st_queue,
                self.type,
                ''
            )
        return ticket_hint

    def process(self, cycle):
        background_statuses = [
            stage_name
            for stage_name in self.config['stages']
            if self.config['stages'][stage_name].get('performer') == 'TACMAN_APPLY_CHANGES'
        ]
        realtime_statuses = [
            stage_name
            for stage_name in self.config['stages']
            if self.config['stages'][stage_name].get('performer') == 'TACMAN_QUEUE'
        ]

        if cycle % self.Parameters.st_period == 0:
            # Find tickets in needed status - this has about 10 seconds delay
            tickets = self.st_helper.get_all_tickets(
                queue=self.Parameters.st_queue,
                tacman_statuses=background_statuses + realtime_statuses,
            )
        else:
            tickets = []

        # Detect hint to changed ticket in first ticket of the queue, as it works much faster than search
        ticket_hint = self.detect_ticket_hint()
        if ticket_hint:
            tickets.append(ticket_hint)

        if self.Parameters.dry_run:
            logging.info('Precess of new Tickets: {} skiped. Task works in Dry Run mode'.format(tickets))
            return

        for ticket in tickets:
            real_status = self.st_helper.get_local_field(ticket, 'tacman_status')
            logging.debug('{} has status {}'.format(ticket, real_status))
            if real_status not in background_statuses + realtime_statuses:
                continue

            for status in realtime_statuses:
                self.process_approve(ticket, status, real_status)

            for status in background_statuses:
                self.process_ticket(ticket, status, real_status)

    def show_started_task(self, title, task_id):
        self.set_info('Started <a href={url}>"{title}" task</a>'.format(
            title=title,
            url=get_sandbox_link(task_id)
        ), do_escape=False)

    def enque_me(self):
        task = TacmanQueue(
            self,
            description='Auto restarted queue ' + self.Parameters.st_queue,
            owner=self.owner,
            dry_run=self.Parameters.dry_run,
            tokens=self.Parameters.tokens,
            use_testing_startrek=self.Parameters.use_testing_startrek,
            st_queue=self.Parameters.st_queue,
            repeats=self.Parameters.repeats,
            sleep_between_repeats=self.Parameters.sleep_between_repeats,
            auto_restart=self.Parameters.auto_restart,
            auto_restart_on_failure=self.Parameters.auto_restart_on_failure,
        )
        task.enqueue()
        self.show_started_task('Auto restarted queue ' + self.Parameters.st_queue, task.id)

    def on_enqueue(self):
        super(TacmanQueue, self).on_enqueue()

        if self.Parameters.st_queue == 'TESTTACCHANGES':
            semaphore_name = 'TACMAN_QUEUE_TEST'
        else:
            semaphore_name = 'TACMAN_QUEUE'
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name=semaphore_name)
            ]
        )

    def init(self):
        self.config = get_config()

        self.last_issue_update = {}

        tokens = self.Parameters.tokens.data()
        self.st_helper = StartrekHelper(
            useragent='sandbox',
            startrek_api_url=TESTING_STARTREK_API_ENDPOINT
                if self.Parameters.use_testing_startrek
                else STARTREK_API_ENDPOINT,
            st_token=tokens['st_token'],
            local_fields_prefix=self.config['queues'][self.Parameters.st_queue]['local_fields_prefix'],
        )
        self.ok_helper = OkHelper(tokens['ok_token'], self.config['approvals'])
        self.control_helper = ControlHelper(tokens['yql_token'])

    def on_execute(self):
        self.init()

        try:
            with self.memoize_stage.process_ticket(commit_on_entrance=True):
                self.set_info('Started listening to ticket changes')
                for cycle in range(self.Parameters.repeats):

                    self.process(cycle)
                    if cycle < self.Parameters.repeats - 1:
                        time.sleep(self.Parameters.sleep_between_repeats)

            if self.Parameters.auto_restart:
                self.enque_me()

        except StopTaskGracefully as e:
            self.set_info('Stopping this task: {}'.format(str(e)))

    def finish(self, prev_status, status):
        if status == 'SUCCESS' or status == 'STOPPED':
            return
        if self.Parameters.auto_restart_on_failure:
            self.enque_me()

    def on_finish(self, prev_status, status):
        self.finish(prev_status, status)

    def on_break(self, prev_status, status):
        self.finish(prev_status, status)
